<?php
/**
 * FUNCTIONS.PHP - Core User Management and Utility Functions
 * ===========================================================
 * 
 * This file contains all essential backend functionality for the Principles website.
 * It handles user authentication, data management, and system utilities.
 * 
 * MAIN CATEGORIES:
 * 1. Translation and Language Functions
 * 2. User Management (Registration, Login, Profile)
 * 3. Data Storage (JSON file operations)
 * 4. Authentication and Authorization
 * 5. User ID System (Unique ID generation and validation)
 * 6. Data Cleanup and Maintenance
 * 
 * IMPORTANT NOTES:
 * - All user data is stored in walkers.json (file-based, no database)
 * - Uses PHP sessions for user authentication
 * - User IDs are 10-digit unique identifiers for security
 * - Moderator permissions are based on user_id (not username)
 * - Includes automatic cleanup of orphaned data
 */

session_start();

// Ensure settings.php is loaded for getText() function
require_once __DIR__ . '/settings.php';

// =============================================================================
// TRANSLATION AND LANGUAGE FUNCTIONS
// =============================================================================

/**
 * Translation function - Main interface for getting translated text
 * 
 * This is an alias for getAppText() from settings.php that provides backward compatibility.
 * It retrieves translated text based on the current language (English or other).
 * 
 * Usage: t('username', 'en') returns "Username"
 *        t('username', 'fr') returns "Nom d'utilisateur"
 * 
 * @param string $key The translation key (e.g., 'username', 'password', 'login')
 * @param string $lang The language code (optional, will auto-detect if not provided)
 * @return string The translated text or the key itself if translation not found
 */
function t($key, $lang = null) {
    // Use getAppText() function from settings.php
    // Renamed to avoid conflict with PHP's built-in gettext()
    if (function_exists('getAppText')) {
        return getAppText($key, $lang);
    }
    // Fallback if getAppText is not available
    return $key;
}

// =============================================================================
// PRINCIPLE TRANSLATION SYSTEM (Optional Feature)
// =============================================================================

/**
 * Principle translations storage
 * 
 * This array stores translations for principle content (titles and descriptions).
 * Currently includes demo content. You can add your own translations here.
 * 
 * Structure:
 * 'English Title' => [
 *     'language_code' => [
 *         'title' => 'Translated Title',
 *         'description' => 'Translated Description'
 *     ]
 * ]
 */
$PRINCIPLE_TRANSLATIONS = [
    'Resilience in Adversity' => [
        'fr' => [
            'title' => 'Résilience dans l\'adversité',
            'description' => 'Quand la vie devient difficile, nous devons nous accrocher plus fort à notre but. La résilience n\'est pas de ne jamais tomber, mais de se relever à chaque fois.'
        ]
    ],
    // Add more principle translations as needed
];

/**
 * Translate principle content to specified language
 * 
 * Looks up the principle title in the translations array and returns
 * the translated version if available. Falls back to original if not found.
 * 
 * @param array $principle The principle array with 'title' and 'description'
 * @param string $lang The target language code (e.g., 'fr', 'es', 'de')
 * @return array The translated principle or original if translation not found
 */
function translate_principle($principle, $lang) {
    global $PRINCIPLE_TRANSLATIONS;
    if ($lang === 'en') return $principle;
    $t = $PRINCIPLE_TRANSLATIONS[$principle['title']] ?? null;
    if ($t && isset($t[$lang])) {
        return [
            'title' => $t[$lang]['title'],
            'description' => $t[$lang]['description'] ?? $principle['description'],
            'created_at' => $principle['created_at']
        ];
    }
    return $principle;
}

// =============================================================================
// USER DATA MANAGEMENT - JSON FILE OPERATIONS
// =============================================================================

/**
 * Get all users from the JSON file
 * 
 * Reads the walkers.json file and returns all user data as an array.
 * This is the primary function for accessing user data in the system.
 * 
 * Returns empty array if file doesn't exist or is corrupted.
 * Each user object contains: user_id, username, email, password, joined, approved
 * 
 * @return array Array of user objects (empty array if no users or file error)
 */
function getUsers() {
    $jsonFile = __DIR__ . '/walkers.json';
    if (file_exists($jsonFile)) {
        $jsonData = file_get_contents($jsonFile);
        $users = json_decode($jsonData, true);
        // Ensure it returns an array even if the file is empty or corrupt
        return is_array($users) ? $users : [];
    }
    return [];
}

/**
 * Save users to the JSON file
 * 
 * Writes the users array back to walkers.json with pretty formatting.
 * This is called after any user data modification (registration, profile updates, deletions).
 * 
 * Uses JSON_PRETTY_PRINT for human-readable formatting in the file.
 * 
 * @param array $users Array of user objects to save
 * @return void
 */
function saveUsers($users) {
    $jsonFile = __DIR__ . '/walkers.json';
    file_put_contents($jsonFile, json_encode($users, JSON_PRETTY_PRINT));
}

// =============================================================================
// USER REGISTRATION AND AUTHENTICATION
// =============================================================================

/**
 * Register a new user in the system
 * 
 * Creates a new user account with the following process:
 * 1. Normalizes username to lowercase
 * 2. Checks if username already exists (case-insensitive)
 * 3. Checks rate limit (max 10 registrations per hour)
 * 4. Generates a unique 10-digit user_id
 * 5. Hashes the password for security
 * 6. Saves user to walkers.json
 * 
 * Rate Limiting: Prevents spam by limiting to 10 new registrations per hour.
 * If limit is exceeded, returns error message.
 * 
 * Returns array with 'success' boolean and 'message' string.
 * On success, also returns normalized 'username' for session storage.
 * 
 * @param string $username The desired username (will be normalized to lowercase)
 * @param string $email The email address (optional, can be empty)
 * @param string $password The plain text password (will be hashed)
 * @return array ['success' => bool, 'username' => string, 'message' => string]
 */
function registerUser($username, $email, $password) {
    $users = getUsers();
    $normalized = strtolower(trim($username));
    
    // Check if username already exists (case-insensitive)
    foreach ($users as $user) {
        if (isset($user['username']) && strtolower($user['username']) === $normalized) {
            return ['success' => false, 'message' => 'Username already exists'];
        }
    }
    
    // Rate limiting: Check if more than 10 users registered in the last hour
    $one_hour_ago = date('Y-m-d H:i:s', strtotime('-1 hour'));
    $recent_registrations = 0;
    foreach ($users as $user) {
        if (isset($user['joined']) && $user['joined'] >= $one_hour_ago) {
            $recent_registrations++;
        }
    }
    
    // Block registration if rate limit exceeded
    if ($recent_registrations >= 10) {
        return ['success' => false, 'message' => 'Registration limit reached. Please try again later.'];
    }
    
    // (No email uniqueness check)
    // Generate unique user_id
    $user_id = generateUniqueUserId();
    
    $newUser = [
        'user_id' => $user_id,
        'username' => $normalized,
        'email' => $email,
        'password' => password_hash($password, PASSWORD_DEFAULT),
        'joined' => date('Y-m-d H:i:s'),
        'approved' => false  // New users require moderator approval
    ];
    
    $users[] = $newUser;
    saveUsers($users);
    
    // Return success with the normalized username
    return ['success' => true, 'username' => $normalized, 'message' => 'Registration successful'];
}


// =============================================================================
// USER LOOKUP FUNCTIONS - Find Users by Different Identifiers
// =============================================================================

/**
 * Get user by username - Primary user lookup function
 * 
 * Searches for a user by their username (case-insensitive).
 * This is the most commonly used lookup function in the system.
 * 
 * Used for: Login, profile viewing, authentication checks
 * 
 * @param string $username The username to search for (case-insensitive)
 * @return array|null User object if found, null if user doesn't exist
 */
function getUserByUsername($username) {
    $users = getUsers();
    $needle = strtolower(trim($username));
    foreach ($users as $user) {
        if (isset($user['username']) && strtolower($user['username']) === $needle) {
            return $user;
        }
    }
    return null;
}


// =============================================================================
// AUTHENTICATION AND AUTHORIZATION
// =============================================================================

/**
 * Check if user is logged in
 * 
 * Simple check to see if a user session exists.
 * Returns true if username is stored in the session.
 * 
 * Used throughout the site to show/hide features and protect pages.
 * 
 * @return bool True if user is logged in, false if guest/anonymous
 */
function isLoggedIn() {
    return isset($_SESSION['username']);
}

/**
 * Check if current user is a moderator
 * 
 * Determines if the logged-in user has moderator privileges.
 * Moderators can approve/reject principles and access special controls.
 * 
 * Process:
 * 1. Checks if user is logged in
 * 2. Gets user's user_id from walkers.json
 * 3. Checks if user_id is in the MODERATORS array in settings.php
 * 
 * Moderator IDs are configured in settings.php under 'MODERATORS'.
 * Uses user_id (not username) for security and consistency.
 * 
 * @return bool True if user is a moderator, false otherwise
 */
function isModerator() {
    if (!isLoggedIn()) return false;
    
    // Get current user to access user_id
    $user = getUserByUsername($_SESSION['username']);
    if (!$user || !isset($user['user_id'])) return false;
    
    // Use the settings system with user_id
    if (function_exists('isModeratorBySettings')) {
        return isModeratorBySettings($user['user_id']);
    }
    
    return false;
}

/**
 * Logout the current user
 * 
 * Destroys the PHP session, effectively logging out the user.
 * After this, isLoggedIn() will return false.
 * 
 * Typically followed by a redirect to the homepage or login page.
 * 
 * @return void
 */
function logoutUser() {
    session_destroy();
}

/**
 * Get default language based on browser settings
 * 
 * Determines which language to display based on browser's Accept-Language header.
 * This is a wrapper for getCurrentLanguage() from settings.php.
 * 
 * Language detection logic:
 * 1. Checks browser's Accept-Language header
 * 2. If it matches 'other_language_code' in settings -> shows other language
 * 3. Otherwise -> shows English
 * 
 * Example: If other_language_code is 'fr' and browser is French -> shows French
 *          If other_language_code is 'fr' and browser is German -> shows English
 * 
 * @return string Language code ('en' for English, 'other' for configured language)
 */
function get_default_lang() {
    // Use the getCurrentLanguage() function from settings.php
    if (function_exists('getCurrentLanguage')) {
        return getCurrentLanguage();
    }
    
    // Fallback if getCurrentLanguage is not available
    if (function_exists('getSetting')) {
        return getSetting('LANGUAGES.default', 'en');
    }
    
    // Final fallback
    return 'en';
} 

// =============================================================================
// USER ID SYSTEM - Unique Identifier Generation and Validation
// =============================================================================

/**
 * Generate a unique 10-digit user_id
 * 
 * Creates a random 10-digit numeric ID that doesn't conflict with existing users.
 * This ID is used for moderator permissions and user identification.
 * 
 * Process:
 * 1. Gets all existing user_ids from walkers.json
 * 2. Generates random 10-digit number
 * 3. Checks if it already exists
 * 4. Repeats until unique ID is found
 * 
 * Why 10 digits: Provides 10 billion possible IDs (very low collision chance)
 * 
 * @return string A unique 10-digit numeric string (e.g., '2970089823')
 */
function generateUniqueUserId() {
    $users = getUsers();
    $existing_ids = array_column($users, 'user_id');
    
    do {
        $user_id = str_pad(rand(1000000000, 9999999999), 10, '0', STR_PAD_LEFT);
    } while (in_array($user_id, $existing_ids));
    
    return $user_id;
}

/**
 * Check if user_id exists in the system
 * 
 * Verifies whether a given user_id belongs to an existing user.
 * Useful for validation before assigning moderator privileges or linking data.
 * 
 * @param string $user_id The 10-digit user_id to check
 * @return bool True if user_id exists, false if not found
 */
function userIdExists($user_id) {
    $users = getUsers();
    foreach ($users as $user) {
        if (isset($user['user_id']) && $user['user_id'] === $user_id) {
            return true;
        }
    }
    return false;
}

/**
 * Get user by user_id
 * 
 * Retrieves a user's complete data using their unique user_id.
 * This is the preferred lookup method for system operations.
 * 
 * More secure than username lookup because user_ids are not publicly visible.
 * Used for: Moderator checks, principle ownership verification, admin operations
 * 
 * @param string $user_id The 10-digit user_id to search for
 * @return array|null User object if found, null if user_id doesn't exist
 */
function getUserByUserId($user_id) {
    $users = getUsers();
    foreach ($users as $user) {
        if (isset($user['user_id']) && $user['user_id'] === $user_id) {
            return $user;
        }
    }
    return null;
}

/**
 * Require login - Redirect to login page if not authenticated
 * 
 * Protection function for pages that require authentication.
 * Call this at the top of any page that should only be accessible to logged-in users.
 * 
 * If user is not logged in, redirects to go.php (login page) with a message.
 * Execution stops after redirect (exit called).
 * 
 * Usage: Place at top of protected pages like profile.php
 * 
 * @return void (exits if not logged in)
 */
function require_login() {
    if (!isset($_SESSION['username'])) {
        header('Location: go.php?message=Please+log+in+to+continue');
        exit;
    }
}

// =============================================================================
// USER AND DATA DELETION - Cleanup Operations
// =============================================================================

/**
 * Delete a user and all their associated principles
 * 
 * Permanently removes a user from the system and cleans up all their data.
 * This is a destructive operation that cannot be undone.
 * 
 * Process:
 * 1. Removes user from walkers.json
 * 2. Calls deletePrinciplesByUserId to remove all their principles
 * 3. Saves updated data to files
 * 
 * Use case: Account deletion, moderation, data cleanup
 * 
 * @param string $user_id The 10-digit user_id of the user to delete
 * @return bool True if user was found and deleted, false if user_id not found
 */
function deleteUser($user_id) {
    $users = getUsers();
    $user_found = false;
    
    // Remove user from users array
    $users = array_filter($users, function($user) use ($user_id, &$user_found) {
        if (isset($user['user_id']) && $user['user_id'] === $user_id) {
            $user_found = true;
            return false; // Remove this user
        }
        return true; // Keep this user
    });
    
    if (!$user_found) {
        return false; // User not found
    }
    
    // Save updated users
    saveUsers(array_values($users));
    
    // Delete all principles by this user
    deletePrinciplesByUserId($user_id);
    
    return true;
}

/**
 * Delete all principles by a specific user ID
 * 
 * Removes all principles authored by a specific user from principles.json.
 * Called automatically when a user is deleted to maintain data integrity.
 * 
 * Process:
 * 1. Loads all principles from principles.json
 * 2. Filters out principles with matching posted_by_user_id
 * 3. Saves the filtered list back to file
 * 4. Returns count of deleted principles
 * 
 * Used by: deleteUser() function, manual cleanup operations
 * 
 * @param string $user_id The 10-digit user_id whose principles to delete
 * @return int Number of principles that were deleted
 */
function deletePrinciplesByUserId($user_id) {
    $principlesFile = __DIR__ . '/principles.json';
    if (!file_exists($principlesFile)) {
        return 0;
    }
    
    $principles = json_decode(file_get_contents($principlesFile), true);
    if (!is_array($principles)) {
        return 0;
    }
    
    $original_count = count($principles);
    
    // Filter out principles by this user
    $principles = array_filter($principles, function($principle) use ($user_id) {
        return !(isset($principle['posted_by_user_id']) && $principle['posted_by_user_id'] === $user_id);
    });
    
    $deleted_count = $original_count - count($principles);
    
    // Save updated principles
    file_put_contents($principlesFile, json_encode(array_values($principles), JSON_PRETTY_PRINT));
    
    return $deleted_count;
}

/**
 * Clean up orphaned principles - Automatic data integrity maintenance
 * 
 * Removes principles whose authors (users) no longer exist in the system.
 * This prevents "Unknown" usernames from appearing on the principles page.
 * 
 * Orphaned principles occur when:
 * - A user is deleted but their principles weren't removed
 * - Data corruption or manual file editing
 * - Migration issues
 * 
 * Process:
 * 1. Loads all principles from principles.json
 * 2. Checks if each principle's posted_by_user_id exists in walkers.json
 * 3. Removes principles with non-existent user_ids
 * 4. Saves cleaned data back to file
 * 
 * Called automatically on principles.php page load for continuous cleanup.
 * 
 * @return int Number of orphaned principles that were removed
 */
function cleanupOrphanedPrinciples() {
    $principlesFile = __DIR__ . '/principles.json';
    if (!file_exists($principlesFile)) {
        return 0;
    }
    
    $principles = json_decode(file_get_contents($principlesFile), true);
    if (!is_array($principles)) {
        return 0;
    }
    
    $original_count = count($principles);
    
    // Filter out principles whose users don't exist
    $principles = array_filter($principles, function($principle) {
        if (!isset($principle['posted_by_user_id'])) {
            return false; // Remove principles without user_id
        }
        return getUserByUserId($principle['posted_by_user_id']) !== null;
    });
    
    $removed_count = $original_count - count($principles);
    
    // Save updated principles
    file_put_contents($principlesFile, json_encode(array_values($principles), JSON_PRETTY_PRINT));
    
    return $removed_count;
} 