<?php
require_once 'functions.php';

// =============================================================================
// LANGUAGE DETECTION AND SETUP
// =============================================================================

// Get current language (auto-detect from browser or use session)
$lang = $_SESSION['lang'] ?? get_default_lang();

// =============================================================================
// LOGOUT HANDLING
// =============================================================================

// Handle logout POST request (destroys session and redirects to homepage)
if (isset($_POST['logout'])) {
    logoutUser();
    header('Location: index.php');
    exit;
}

// =============================================================================
// INITIALIZE FORM STATE VARIABLES
// =============================================================================

$error = '';                // Error message to display (empty if no error)
$success_message = '';      // Success message for new user detection
$show_confirm = false;      // Whether to show password confirmation field
$pending_username = '';     // Username waiting for confirmation
$pending_password = '';     // Password waiting for confirmation
$stored_username = '';      // Username to pre-fill if validation fails

// =============================================================================
// STEP 2: ACCOUNT CONFIRMATION (New Users Only)
// =============================================================================

// Handle account confirmation - This is the second step for new users
if (isset($_POST['confirm_account'])) {
    $confirm_password = $_POST['confirm_password'] ?? '';
    $username = $_POST['pending_username'] ?? '';
    $password = $_POST['pending_password'] ?? '';
    
    // Verify that both passwords match
    if ($confirm_password !== $password) {
        $error = 'Passwords do not match. Please try again.';
        $show_confirm = true;
        $pending_username = $username;
        $pending_password = $password;
    } else {
        // Passwords match - Create the account
        $registration_result = registerUser($username, '', $password);
        
        if ($registration_result['success']) {
            // Store normalized username (lowercased) in session
            $_SESSION['username'] = $registration_result['username'] ?? strtolower($username);
            // Redirect to profile page
            header('Location: profile');
            exit;
        } else {
            // Registration failed (username might have been taken in the meantime)
            $error = $registration_result['message'] ?? 'Registration failed. Please try again.';
        }
    }
}

// =============================================================================
// STEP 1: INITIAL LOGIN/SIGNUP ATTEMPT
// =============================================================================

// Handle initial form submission - Determines if user is logging in or signing up
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['confirm_account'])) {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $stored_username = $username;  // Store for re-displaying if error occurs

    // Validation: Check if both fields are filled
    if (empty($username) || empty($password)) {
        $error = t('error_fill_fields', $lang);
    } 
    // Validation: Check minimum password length
    elseif (strlen($password) < 6) {
        $error = t('error_password_length', $lang);
    } 
    // Both fields valid - Check if user exists
    else {
        $user = getUserByUsername($username);
        
        if ($user) {
            // EXISTING USER PATH: Username found in database - Attempt login
            if (password_verify($password, $user['password'])) {
                // Password correct - Login successful
                $_SESSION['username'] = $user['username'];
                header('Location: profile');
                exit;
            } else {
                // Password incorrect - Show error
                $error = 'Incorrect password for existing username.';
            }
        } else {
            // NEW USER PATH: Username not found - Start registration process
            $success_message = 'New username detected. Please re-enter your password to confirm the creation of your account.';
            $show_confirm = true;           // Show confirmation field
            $pending_username = $username;  // Store username for next step
            $pending_password = $password;  // Pre-fill password for confirmation
        }
    }
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo t('login_signup_topbar', $lang); ?></title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:ital,wght@0,100..900;1,100..900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="style.css" />
    <link rel="stylesheet" href="style.php" />
    <style>
    /* Page-specific styles for authentication form */
        .auth-container {
            /* min-height: calc(100vh - 120px); Adjusted for nav and lang bar */
            display: flex;
            align-items: center;
            justify-content: center;
            /* background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%); */
            /* padding: 20px; */
        }
        .auth-box {
            background: <?php echo getSetting('PAGES.signup.container_background_color', getSetting('COLORS.signup_container_background_color', '#ffffff')); ?>;
            padding: 40px;
            border-radius: 30px;
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
            width: 100%;
            max-width: 500px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        .form-group input {
            width: 100%;
            padding: 10px;
            border: 3px solid <?php echo getSetting('PAGES.signup.input_border_color', '#dee2e6'); ?>;
            border-radius: 18px;
            font-size: 1.25rem;
            background: <?php echo getSetting('PAGES.signup.input_background_color', '#f8f9fa'); ?>;
            color: <?php echo getSetting('PAGES.signup.input_text_color', '#333333'); ?>;
            transition: border-color 0.3s;
        }
        .form-group input:focus {
            border-color: <?php echo getSetting('PAGES.signup.input_border_color', '#dee2e6'); ?>;
            outline: none;
        }
        .btn-primary {
            width: 100%;
            padding: 10px 24px;
            background: <?php echo getSetting('PAGES.signup.button_background_color', '#f9a1b0'); ?>;
            color: <?php echo getSetting('PAGES.signup.button_text_color', '#ffffff'); ?>;
            border: none;
            border-radius: 50px;
            font-size: 1rem;
            cursor: pointer;
            margin-top: 20px;
            transition: background-color 0.3s;
        }
        
        .error-message {
            background: #ffebee;
            color: #c62828;
            padding: 10px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        .nav-container {
            width: 100%;
            max-width: 900px;
            margin: 0 auto;
        }
        .nav-left {
            display: flex;
            width: 100%;
            gap: 0;
        }
        @media screen and (min-width: 768px) {
            .signup-page {
                /* padding-right: 15px; */
            }
        }
    </style>
</head>
<body class="signup-page">
<nav class="navbar">
    <div class="nav-container nav-flex">
        <div class="nav-left">
            <a href="index.php" class="nav-button"><?php echo t('prescription', $lang); ?></a>
            <a href="principles.php" class="nav-button"><?php echo t('principles', $lang); ?></a>
            <a href="walkers.php" class="nav-button"><?php echo t('walkers', $lang); ?></a>
            <?php if (isLoggedIn()): ?>
                <a href="profile.php" class="nav-button profile-btn"><?php echo t('profile', $lang); ?></a>
            <?php else: ?>
                <a href="go.php" class="nav-button login-btn"><?php echo t('login_signup_topbar', $lang); ?></a>
            <?php endif; ?>
        </div>
    </div>
</nav>

<main class="container">
    <div class="auth-container">
        <div class="auth-box">
            <!-- Authentication Form: Handles both login and registration -->
            <form class="auth-form" method="POST" action="go" id="authForm">
                
                <!-- Username Field: Always visible, locked during confirmation step -->
                <div class="form-group">
                    <input type="text" id="username" name="username" placeholder="<?php echo t('username', $lang); ?>" 
                           value="<?php echo htmlspecialchars($show_confirm ? $pending_username : $stored_username); ?>" 
                           <?php echo $show_confirm ? 'readonly style="background: #f5f5f5;"' : ''; ?> 
                           required maxlength="32">
                </div>
                
                <div class="form-group">
                    <input type="password" id="password" name="password" 
                           placeholder="<?php echo t('password', $lang); ?>" 
                           value="<?php echo $show_confirm ? htmlspecialchars($pending_password) : ''; ?>"
                           <?php echo $show_confirm ? 'readonly style="background: #f5f5f5;"' : ''; ?>
                           required minlength="6">
                </div>
                
                <?php if ($error): ?>
                    <div class="error-message" style="margin-top: 10px; margin-bottom: 15px;"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>
                
                <?php if ($success_message): ?>
                    <div style="background: #e8f5e9; color: #2e7d32; padding: 10px; border-radius: 8px; margin-top: 10px; margin-bottom: 15px; text-align: center;">
                        <?php echo htmlspecialchars($success_message); ?>
                    </div>
                <?php endif; ?>
                
                <!-- Confirmation Step: Only shown for new users -->
                <?php if ($show_confirm): ?>
                    <!-- Confirm Password Field: New users must re-enter password -->
                    <div class="form-group">
                        <input type="password" id="confirm_password" name="confirm_password" placeholder="Re-enter password to confirm" required>
                        <!-- Real-time validation indicator: Shows checkmark or X -->
                        <div id="password-match-message" style="margin-top: 5px; font-size: 0.9rem;"></div>
                    </div>
                    <!-- Hidden fields: Pass username and password to confirmation handler -->
                    <input type="hidden" name="pending_username" value="<?php echo htmlspecialchars($pending_username); ?>">
                    <input type="hidden" name="pending_password" value="<?php echo htmlspecialchars($pending_password); ?>">
                    <!-- Confirm Button: Disabled until passwords match -->
                    <button type="submit" name="confirm_account" class="btn btn-primary" id="confirmBtn" disabled style="opacity: 0.5; cursor: not-allowed;">Confirm & Create Account</button>
                <?php else: ?>
                    <!-- Login/Signup Button: For initial form submission -->
                    <button type="submit" class="btn btn-primary"><?php echo t('login_signup_login_page', $lang); ?></button>
                <?php endif; ?>
            </form>
        </div>
    </div>
</main>

<?php if ($show_confirm): ?>
<!-- Real-time Password Matching Validation JavaScript -->
<!-- This script only loads when showing the confirmation step for new users -->
<script>
// =============================================================================
// REAL-TIME PASSWORD MATCHING VALIDATION
// =============================================================================
// 
// This JavaScript provides instant feedback as the user types in the confirm
// password field, showing whether passwords match with visual indicators.
// The submit button is disabled until passwords match perfectly.

// Get DOM elements for password validation
const passwordField = document.getElementById('password');           // Original password (locked/readonly)
const confirmPasswordField = document.getElementById('confirm_password');  // Confirmation password field
const confirmBtn = document.getElementById('confirmBtn');            // Submit button
const matchMessage = document.getElementById('password-match-message');    // Validation message display

/**
 * Check if passwords match and update UI accordingly
 * 
 * This function runs every time the user types in the confirm password field.
 * It provides three states:
 * 1. Empty: No message, button disabled
 * 2. Match: Green checkmark, button enabled
 * 3. No Match: Red X, button disabled
 */
function checkPasswordMatch() {
    const password = passwordField.value;
    const confirmPassword = confirmPasswordField.value;
    
    // State 1: Confirm field is empty
    if (confirmPassword === '') {
        matchMessage.textContent = '';
        matchMessage.style.color = '';
        confirmBtn.disabled = true;
        confirmBtn.style.opacity = '0.5';
        confirmBtn.style.cursor = 'not-allowed';
    } 
    // State 2: Passwords match - Enable submission
    else if (password === confirmPassword) {
        matchMessage.textContent = '✓';  // Green checkmark
        matchMessage.style.color = '#2e7d32';
        confirmBtn.disabled = false;
        confirmBtn.style.opacity = '1';
        confirmBtn.style.cursor = 'pointer';
    } 
    // State 3: Passwords don't match - Keep disabled
    else {
        matchMessage.textContent = '✗';  // Red X
        matchMessage.style.color = '#c62828';
        confirmBtn.disabled = true;
        confirmBtn.style.opacity = '0.5';
        confirmBtn.style.cursor = 'not-allowed';
    }
}

// Attach validation to input events (runs as user types)
confirmPasswordField.addEventListener('input', checkPasswordMatch);
confirmPasswordField.addEventListener('keyup', checkPasswordMatch);

// Final validation before form submission (safety check)
document.getElementById('authForm').addEventListener('submit', function(e) {
    if (passwordField.value !== confirmPasswordField.value) {
        e.preventDefault();  // Stop form submission
        matchMessage.textContent = '✗';
        matchMessage.style.color = '#c62828';
    }
});
</script>
<?php endif; ?>

</body>
</html> 