<?php

/**
 * Load content from language-specific files
 * 
 * Determines which content file to load based on the current language.
 * English uses fpeng.txt, all other languages use fpother.txt.
 * 
 * Process:
 * 1. Determines filename based on language (fpeng.txt or fpother.txt)
 * 2. Checks if file exists
 * 3. Reads file contents
 * 4. Parses content using parseCustomContent function
 * 5. Returns array with title and processed HTML content
 * 
 * Returns error messages if file is missing or unreadable.
 * 
 * @param string $language Language code ('en' for English, 'other' for configured language)
 * @return array Array with 'title' and 'content' keys
 */
function loadContentFromFile($language) {
    $filename = ($language === 'en') ? 'fpeng.txt' : 'fpother.txt';
    $filepath = __DIR__ . '/' . $filename;
    
    if (!file_exists($filepath)) {
        return [
            'title' => 'Content Not Found',
            'content' => 'Please create ' . $filename . ' to add content.'
        ];
    }
    
    $raw_content = file_get_contents($filepath);
    if ($raw_content === false) {
        return [
            'title' => 'Error Loading Content',
            'content' => 'Could not read ' . $filename . '.'
        ];
    }
    
    return parseCustomContent($raw_content);
}

/**
 * Parse custom tags and convert them to HTML
 * 
 * Extracts the page title and processes custom content tags into proper HTML.
 * This is the main content parsing function.
 * 
 * Process:
 * 1. Extracts title from "title=" line (first line of content file)
 * 2. Removes title line from content
 * 3. Normalizes newlines and whitespace
 * 4. Calls processCustomTags to convert tags to HTML
 * 5. Returns array with title and processed HTML content
 * 
 * @param string $raw_content Raw text content from fpeng.txt or fpother.txt
 * @return array Array with 'title' and 'content' keys
 */
function parseCustomContent($raw_content) {
    $title = '';
    $content = $raw_content;

    // Extract title from any line that starts with title=
    if (preg_match('/^title=(.*)$/m', $raw_content, $m)) {
        $title = trim($m[1]);
        // Remove the title line from the content
        $content = preg_replace('/^title=.*$/m', '', $raw_content, 1);
    }

    // Normalize newlines and trim leading/trailing whitespace
    $content = trim($content);

    // Process the full content at once so tags like [par]...[/par] can span multiple lines
    $processed = processCustomTags($content);

    return [
        'title' => $title,
        'content' => $processed,
    ];
}

/**
 * Process custom tags and convert to HTML
 */
function processCustomTags($text) {
    // =========================================================================
    // TAG 1: [IMG] - Image Embedding (Local and External)
    // =========================================================================
    // Converts [img]url[/img] to centered, responsive image with styling.
    // Supports both:
    // - External URLs: [img]https://example.com/photo.jpg[/img]
    // - Local files: [img]myimage.jpg[/img] (from root directory)
    $text = preg_replace_callback(
        '/\[img\](.*?)\[\/img\]/i',
        function($matches) {
            $url = trim($matches[1]);
            // Check if it's a full URL (http/https) or a local file
            $is_external = filter_var($url, FILTER_VALIDATE_URL);
            $is_local = !$is_external && !empty($url);
            
            if ($is_external || $is_local) {
                // Generate centered image with rounded corners and shadow
                return '<div class="content-image-container" style="text-align: center; margin: 30px 0;">
                    <img src="' . htmlspecialchars($url) . '" alt="Content Image" style="text-align: center;max-width: 100%; height: auto; border-radius: 12px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                </div>';
            }
            return '';  // Invalid URL - return empty string
        },
        $text
    );
    
    // =========================================================================
    // TAG 2: [IMGNE] - Image Embedding (No Effects)
    // =========================================================================
    // Converts [imgne]url[/imgne] to centered, responsive image WITHOUT styling effects.
    // Perfect for transparent PNG images that should blend with the background.
    // Supports both:
    // - External URLs: [imgne]https://example.com/photo.png[/imgne]
    // - Local files: [imgne]myimage.png[/imgne] (from root directory)
    $text = preg_replace_callback(
        '/\[imgne\](.*?)\[\/imgne\]/i',
        function($matches) {
            $url = trim($matches[1]);
            // Check if it's a full URL (http/https) or a local file
            $is_external = filter_var($url, FILTER_VALIDATE_URL);
            $is_local = !$is_external && !empty($url);
            
            if ($is_external || $is_local) {
                // Generate centered image WITHOUT effects (no shadows, no rounded corners)
                return '<div class="content-image-container-no-effects" style="text-align: center; margin: 30px 0;">
                    <img src="' . htmlspecialchars($url) . '" alt="Content Image" style="max-width: 100%; height: auto;border-radius: 12px;">
                </div>';
            }
            return '';  // Invalid URL - return empty string
        },
        $text
    );
    
    // =========================================================================
    // TAG 3: [DWN] - Download Button
    // =========================================================================
    // Converts [dwn]file|text[/dwn] to styled download button.
    // Syntax: [dwn]filename.zip|Button Text[/dwn]
    // Example: [dwn]source.zip|Download Source Code[/dwn]
    $text = preg_replace_callback(
        '/\[dwn\](.*?)\[\/dwn\]/i',
        function($matches) {
            $download_data = trim($matches[1]);
            $parts = explode('|', $download_data, 2);  // Split at first pipe character
            
            if (count($parts) >= 2) {
                $file = trim($parts[0]);        // File path (e.g., source.zip)
                $link_text = trim($parts[1]);   // Button text to display
                
                // Get download button color from settings (customizable per page)
                require_once 'settings.php';
                $btn_color = getSetting('PAGES.prescription.download_button_color', '#8FD3FF');
                
                // Generate centered download button with custom styling
                return '<div class="content-download-container" style="text-align: center; margin: 30px 0;">
                    <a href="' . htmlspecialchars($file) . '" class="download-btn" download style="display: inline-block; background: ' . $btn_color . '; color: white; padding: 10px 24px; text-decoration: none; border-radius: 20px; font-weight: 600; font-size: 1rem; text-align: center; transition: all 0.3s ease; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                        ' . htmlspecialchars($link_text) . '
                    </a>
                </div>';
            }
            return '';  // Invalid format - return empty string
        },
        $text
    );
    
    // =========================================================================
    // TAG 4: [PAR] - Paragraph Formatting with Nested Bold Support
    // =========================================================================
    // Converts [par]content[/par] to properly formatted paragraph.
    // Supports nested [bld] tags for bold text within paragraphs.
    // Example: [par]This is [bld]important[/bld] text.[/par]
    // 
    // Processing order is critical:
    // 1. First convert [bld] to <strong>
    // 2. Then escape the text for security
    // 3. Finally unescape the <strong> tags so they render as HTML
    $text = preg_replace_callback(
        '/\[par\](.*?)\[\/par\]/is',  // 'is' flag allows multi-line content
        function($matches) {
            $paragraph_content = trim($matches[1]);
            // Step 1: Process any nested [bld] tags within the paragraph FIRST
            $paragraph_content = preg_replace('/\[bld\](.*?)\[\/bld\]/i', '<strong>$1</strong>', $paragraph_content);
            // Step 2: Safely escape the text for security (prevents XSS)
            $paragraph_content = htmlspecialchars($paragraph_content, ENT_QUOTES, 'UTF-8', false);
            // Step 3: Unescape the <strong> tags we created so they render as HTML
            $paragraph_content = str_replace(['&lt;strong&gt;', '&lt;/strong&gt;'], ['<strong>', '</strong>'], $paragraph_content);
            return '<p>' . $paragraph_content . '</p>';
        },
        $text
    );
    
    // =========================================================================
    // TAG 5: [BLD] - Bold Text (Standalone)
    // =========================================================================
    // Converts [bld]text[/bld] to bold text outside of paragraphs.
    // For bold text INSIDE paragraphs, it's processed within [par] tag handler above.
    $text = preg_replace(
        '/\[bld\](.*?)\[\/bld\]/i',
        '<strong>$1</strong>',
        $text
    );
    
    // =========================================================================
    // PLAIN TEXT HANDLING
    // =========================================================================
    // If content has no HTML tags and no custom tags, escape it for security.
    // This handles edge cases where content is just plain text.
    if (!empty(trim($text)) && strpos($text, '<') === false && strpos($text, '[') === false) {
        $text = htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
    }
    
    return $text;
}

/**
 * Get processed content for the current language
 * 
 * Main entry point for getting prescription page content.
 * Auto-detects language if not specified.
 * 
 * This function:
 * 1. Determines current language (auto-detect or use provided)
 * 2. Calls loadContentFromFile to get and parse content
 * 3. Returns processed HTML ready for display
 * 
 * Used by: index.php (prescription page) to display content
 * 
 * @param string $language Optional language code (auto-detects if null)
 * @return array Array with 'title' and 'content' keys containing processed HTML
 */
function getProcessedContent($language = null) {
    if ($language === null) {
        // Get language from existing system
        require_once 'functions.php';
        $language = getCurrentLanguage();
    }
    
    return loadContentFromFile($language);
}

/**
 * Add CSS styles for the custom content elements
 * 
 * Generates dynamic CSS for prescription page content based on settings.
 * This allows font customization without editing CSS files.
 * 
 * What it customizes:
 * - Font family for all prescription content
 * - Title font size
 * - Paragraph font size
 * - Strong (bold) text font
 * 
 * Font settings come from:
 * PAGES.prescription.font_family
 * PAGES.prescription.title_font_size
 * PAGES.prescription.paragraph_font_size
 * 
 * Returns inline style tag to be injected in page head.
 * 
 * Used by: index.php to apply custom prescription page styling
 * 
 * @return string HTML style tag with generated CSS
 */
function getContentStyles() {
    // Get font settings from settings.php
    require_once 'settings.php';
    $font_family = getSetting('PAGES.prescription.font_family', 'Comic Neue, cursive');
    $title_font_size = getSetting('PAGES.prescription.title_font_size', '2.5rem');
    $paragraph_font_size = getSetting('PAGES.prescription.paragraph_font_size', '1.1rem');
    
    return '
    <style>
        .prescription-content {
            font-family: ' . $font_family . ';
        }
        .prescription-content .landing-title {
            font-size: ' . $title_font_size . ';
            font-family: ' . $font_family . ';
        }
        .prescription-content p {
            font-size: ' . $paragraph_font_size . ';
            line-height: 1.6;
            margin-bottom: 1em;
            font-family: ' . $font_family . ';
        }
        .prescription-content strong {
            font-weight: 600;
            font-family: ' . $font_family . ';
        }
    </style>';
}

?>
