<?php
// Enable error reporting for debugging (can be disabled in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Load required configuration and utility files
require_once 'functions.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// =============================================================================
// AJAX PROFILE FIELD UPDATES HANDLER
// =============================================================================

// Handle AJAX profile field updates (currently not used, but available for future)
if (isset($_POST['field']) && isset($_POST['value'])) {
    header('Content-Type: application/json');
    
    if (!isLoggedIn()) {
        echo json_encode(['status' => 'error', 'message' => t('error_not_logged_in', $lang)]);
        exit;
    }
    
    $allowed_fields = ['email', 'password'];
    $field = $_POST['field'] ?? '';
    $value = $_POST['value'] ?? '';
    
    if (!in_array($field, $allowed_fields)) {
        echo json_encode(['status' => 'error', 'message' => t('error_invalid_field', $lang)]);
        exit;
    }
    
    $username = $_SESSION['username'];
    $user = getUserByUsername($username);
    if (!$user) {
        echo json_encode(['status' => 'error', 'message' => t('error_user_not_found', $lang)]);
        exit;
    }
    
    // Direct update without moderation
    $users = getUsers();
    foreach ($users as &$u) {
        if ($u['username'] === $username) {
            if ($field === 'password') {
                $u['password'] = password_hash($value, PASSWORD_DEFAULT);
            } else {
                $u[$field] = $value;
            }
            break;
        }
    }
    saveUsers($users);
    echo json_encode(['status' => 'success', 'message' => '']);
    exit;
}

// =============================================================================
// LOGOUT HANDLING
// =============================================================================

// Handle logout POST request with smart redirect to public profile view
if (isset($_POST['logout'])) {
    $current_user = $_SESSION['username'] ?? null;
    logoutUser();  // Destroy session
    // Redirect to the same profile in public view (so user can still see the profile)
    if ($current_user) {
        header('Location: profile.php?user=' . urlencode($current_user));
    } else {
        header('Location: index.php');
    }
    exit;
}

// =============================================================================
// LANGUAGE AND USER SETUP
// =============================================================================

// Get current language (auto-detect from browser or use session)
$lang = get_default_lang();

// Determine which profile to display (from URL or logged-in user)
$username = $_GET['user'] ?? ($_SESSION['username'] ?? null);
if (!$username) {
    // No user specified and not logged in - redirect to login
    header('Location: go.php?message=' . urlencode(t('error_login_required', $lang)));
    exit;
}

// Load user data from walkers.json
$user = getUserByUsername($username);
if (!$user) {
    // User doesn't exist - log out and redirect to login
    session_unset();
    session_destroy();
    header('Location: go.php?message=' . urlencode(t('error_login_again', $lang)));
    exit;
}

// Determine if viewing own profile or someone else's
$is_own_profile = isset($_SESSION['username']) && $_SESSION['username'] === $user['username'];

// Security: Prevent unauthorized profile editing
if (!$is_own_profile && isset($_POST['username'])) {
    header('Location: go.php?message=' . urlencode(t('error_edit_permission', $lang)));
    exit;
}

// =============================================================================
// LOAD AND FILTER USER'S PRINCIPLES
// =============================================================================

// Load all principles from JSON file
$principlesFile = __DIR__ . '/principles.json';
$principles = file_exists($principlesFile) ? json_decode(file_get_contents($principlesFile), true) : [];
if (!is_array($principles)) {
    $principles = [];
}

// Filter principles based on profile ownership
if ($is_own_profile) {
    // Own profile: Show all principles (approved and pending)
    $user_principles = array_values(array_filter($principles, function($p) use ($user) {
        return isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id'];
    }));
} else {
    // Other's profile: Show only approved principles
    $user_principles = array_values(array_filter($principles, function($p) use ($user) {
        return isset($p['posted_by_user_id']) && 
               $p['posted_by_user_id'] === $user['user_id'] && 
               isset($p['approved']) && 
               $p['approved'] === true;
    }));
}

// Sort principles by creation date (newest first)
usort($user_principles, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// =============================================================================
// PROFILE UPDATE HANDLER (Email and Password)
// =============================================================================

$update_message = '';
// Handle profile update form submission (email and/or password change)
if (isset($_POST['email']) && $is_own_profile) {
    // Remove all spaces from email and limit to 100 characters
    $new_email = substr(preg_replace('/\s+/', '', $_POST['email'] ?? ''), 0, 100);
    $new_email = trim($new_email);
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // Load all users and update the current user's info
    $users = getUsers();
    foreach ($users as &$u) {
        if ($u['username'] === $user['username']) {
            // Update email
            $u['email'] = $new_email;
            
            // Update password if provided
            if (!empty($new_password)) {
                // Validate password: must match confirmation and be at least 6 characters
                if ($new_password === $confirm_password && strlen($new_password) >= 6) {
                    $u['password'] = password_hash($new_password, PASSWORD_DEFAULT);
                } else {
                    $update_message = t('error_passwords_mismatch_short', $lang);
                    break;
                }
            }
            break;
        }
    }
    
    // Save updated users back to walkers.json
    saveUsers($users);
    
    // Redirect to refresh the page with updated data
    header('Location: profile.php?user=' . urlencode($user['username']));
    exit;
}

// =============================================================================
// ADD NEW PRINCIPLE HANDLER
// =============================================================================

$principle_message = '';
// Handle new principle submission (only for own profile)
if (isset($_POST['add_principle']) && $is_own_profile) {
    $title = trim($_POST['principle_title'] ?? '');
    $desc = trim($_POST['principle_desc'] ?? '');
    
    // Normalize multiple consecutive spaces to single space (if enabled in settings)
    if (getSetting('PRINCIPLE_SETTINGS.normalize_spaces', true)) {
        $title = preg_replace('/\s+/', ' ', $title);
        $desc = preg_replace('/\s+/', ' ', $desc);
    }
    
    // Count words in title
    $titleWords = preg_split('/\\s+/', $title, -1, PREG_SPLIT_NO_EMPTY);
    
    // Remove line breaks from description (principles don't support line breaks)
    $desc = str_replace(["\r\n", "\r", "\n"], ' ', $desc);
    
    // Get customizable limits and labels from settings
    $titleWordLimit = getSetting('PRINCIPLE_SETTINGS.title_word_limit', 1);
    $descCharLimit = getSetting('PRINCIPLE_SETTINGS.description_char_limit', 777);
    $titleLabel = $lang === 'other' ? getSetting('PRINCIPLE_SETTINGS.title_label_other', 'Principe') : getSetting('PRINCIPLE_SETTINGS.title_label_eng', 'Principle');
    $descLabel = $lang === 'other' ? getSetting('PRINCIPLE_SETTINGS.description_label_other', 'Description') : getSetting('PRINCIPLE_SETTINGS.description_label_eng', 'Description');
    
    // Enforce maximum 10 principles per user (hard limit)
    $existing_count = count(array_filter($all_principles ?? (file_exists($principlesFile) ? json_decode(file_get_contents($principlesFile), true) : []), function($p) use ($user) {
        return isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id'];
    }));
    
    if ($existing_count >= 10) {
        // User has reached maximum principles
        $principle_message = 'You have reached the maximum of 10 principles.';
    } elseif ($title && $desc && mb_strlen($desc) <= $descCharLimit && count($titleWords) <= $titleWordLimit) {
        // Validation passed - add the principle
        $all_principles = file_exists($principlesFile) ? json_decode(file_get_contents($principlesFile), true) : [];
        if (!is_array($all_principles)) {
            $all_principles = [];
        }

        // Create new principle object
        $all_principles[] = [
            'id' => uniqid('p_'),
            'title' => $title,
            'description' => $desc,
            'posted_by_user_id' => $user['user_id'],
            'approved' => false,  // All new principles start as pending moderation
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Save to principles.json
        $result = file_put_contents($principlesFile, json_encode($all_principles, JSON_PRETTY_PRINT));
        if ($result === false) {
            // File write failed - display error message
            die("FATAL ERROR: Could not write to principles.json. This is a file permission issue. Please double-check that your web server (Apache) has write permissions for the project folder.");
        }

        // Redirect to refresh the page with new principle
        header('Location: profile.php?user=' . urlencode($user['username']));
        exit;
    } else {
        // Validation failed - show error message
        $wordLabel = $titleWordLimit === 1 ? t('word', $lang) : t('words', $lang);
        $charactersLabel = t('characters', $lang);
        $principle_message = "{$titleLabel} must be {$titleWordLimit} {$wordLabel} only and {$descLabel} under {$descCharLimit} {$charactersLabel} with no line breaks.";
    }
}

// =============================================================================
// EDIT AND DELETE PRINCIPLE HANDLERS
// =============================================================================

// Handle principle edit/delete actions (only for own principles)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['principle_action'])) {
    
    // DELETE PRINCIPLE
    if ($_POST['principle_action'] === 'delete' && isset($_POST['principle_id'])) {
        $principle_id_to_delete = $_POST['principle_id'];
        
        // Filter out the principle to delete (only if user owns it)
        $principles = array_filter($principles, function($p) use ($principle_id_to_delete, $user) {
            // Keep all principles except the one being deleted (if owned by user)
            if ($p['id'] === $principle_id_to_delete) {
                return !(isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id']);
            }
            return true;
        });
        
        // Save updated principles list
        file_put_contents($principlesFile, json_encode(array_values($principles), JSON_PRETTY_PRINT));
        header('Location: profile.php?user=' . urlencode($user['username']));
        exit;
    }
    
    // EDIT PRINCIPLE
    if ($_POST['principle_action'] === 'edit' && isset($_POST['principle_id'], $_POST['edit_title'], $_POST['edit_desc'])) {
        foreach ($principles as &$p) {
            if ($p['id'] === $_POST['principle_id']) {
                // Verify user owns this principle (security check)
                if (isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id']) {
                    $edit_title = trim($_POST['edit_title']);
                    $edit_desc = trim($_POST['edit_desc']);
                    
                    // Normalize spaces if enabled
                    if (getSetting('PRINCIPLE_SETTINGS.normalize_spaces', true)) {
                        $edit_title = preg_replace('/\s+/', ' ', $edit_title);
                        $edit_desc = preg_replace('/\s+/', ' ', $edit_desc);
                    }
                    
                    // Count words in title
                    $edit_titleWords = preg_split('/\\s+/', $edit_title, -1, PREG_SPLIT_NO_EMPTY);
                    
                    // Remove line breaks from description
                    $edit_desc = str_replace(["\r\n", "\r", "\n"], ' ', $edit_desc);
                    
                    // Validate against limits
                    $titleWordLimit = getSetting('PRINCIPLE_SETTINGS.title_word_limit', 1);
                    $descCharLimit = getSetting('PRINCIPLE_SETTINGS.description_char_limit', 777);
                    
                    if (count($edit_titleWords) <= $titleWordLimit && mb_strlen($edit_desc) <= $descCharLimit) {
                        // Update principle
                        $p['title'] = $edit_title;
                        $p['description'] = $edit_desc;
                        $p['approved'] = false;  // Reset to pending moderation after edit
                    }
                }
                break;
            }
        }
        
        // Save updated principles
        file_put_contents($principlesFile, json_encode(array_values($principles), JSON_PRETTY_PRINT));
        header('Location: profile.php?user=' . urlencode($user['username']));
        exit;
    }
}

// Reload and re-filter user principles after any add/edit/delete operations
$user_principles = array_values(array_filter($principles, function($p) use ($user) {
    return isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id'];
}));

// Re-sort principles by creation date (newest first)
usort($user_principles, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// =============================================================================
// UTILITY DATA AND FUNCTIONS
// =============================================================================

/**
 * Obfuscate email address for display
 * 
 * Masks the email to show only first 2 characters of username and domain.
 * Example: "john.doe@example.com" becomes "jo******@ex******.com"
 * 
 * @param string $email The email address to obfuscate
 * @return string The obfuscated email address
 */
function obfuscate_email($email) {
    if (!$email) return '';
    $parts = explode('@', $email);
    if (count($parts) !== 2) return $email;
    $name = substr($parts[0], 0, 2) . str_repeat('*', max(0, strlen($parts[0])-2));
    $domain = explode('.', $parts[1]);
    $domain_main = substr($domain[0], 0, 2) . str_repeat('*', max(0, strlen($domain[0])-2));
    $domain_ext = isset($domain[1]) ? $domain[1] : '';
    return $name . '@' . $domain_main . (strlen($domain_ext) ? ('.' . $domain_ext) : '');
}

// =============================================================================
// UI STATE VARIABLES
// =============================================================================

// Handle success messages from redirects (displayed at top of page)
$success_message = $_GET['message'] ?? '';
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - <?php echo htmlspecialchars($user['username']); ?></title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:ital,wght@0,100..900;1,100..900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="style.css" />
    <link rel="stylesheet" href="style.php" />
    <style>
        .profile-main {
            display: flex;
            width: 100%;
            flex-flow: column;
            gap: 25px;
            max-width: 400px;
            margin: 10px auto 0 auto;
            flex-wrap: wrap;
        }
        .profile-info-block {
            position: relative;
            height: 265px;
            /* flex: 0 0 320px; */
            background: #fff;
            border-radius: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.08);
            padding: 15px 0 0px 20px;
            min-width: 280px;
            max-width: 100%;
        }
        .profile-info-block-inner {
            display: flex;
            height: 100%;
            padding-right: 20px;
            padding-top: 10px;
            justify-content: space-between;
            flex-flow: row-reverse;
        }
        form#profileEditForm {
            width: 100%;
        }
        #editProfileButtons {
            display: flex;
            margin-bottom: 15px;
            padding-right: 0px;
            justify-content: end;
        }
        .profile-page.viewing-own-profile .profile-info-block {
            padding-top: 0;
        }
        .profile-page.viewing-other-profile .principle-header{
            margin-top: 0;
        }
        .profile-page.viewing-other-profile .principle-meta{
            position: unset;
        }
        .profile-username {
            font-size: 2rem;
            margin-bottom: 10px;
        }
        .profile-meta {
            color: #888;
            font-size: 1rem;
            margin-bottom: 10px;
        }
        .profile-info-form label {
            display: block;
            margin-bottom: 4px;
            color: #555;
            font-weight: 500;
            margin-top: 12px;
        }
        .profile-info-form input, .profile-info-form select {
            width: 100%;
            padding: 10px;
            border-radius: 8px;
            border: 1px solid #eee;
            margin-bottom: 4px;
            font-size: 1.1rem;
        }
        .profile-info-form .save-btn {
            background: #7dcf95;
            color: #fff;
            border: none;
            border-radius: 8px;
            padding: 10px 24px;
            font-size: 1.1rem;
            margin-top: 18px;
            cursor: pointer;
            transition: background 0.2s;
        }
        
        .profile-info-form .readonly {
            background: #f7f7f7;
            color: #aaa;
        }
        .profile-info-form .joined-date {
            margin-top: 10px;
            color: #888;
            font-size: 1rem;
        }
        /* #profile-value-email strong{
            margin-top: -21px; 
            
        } */
        td#profile-value-username{
            width: 55%;
        }
        #profile-value-email{
            position: relative;
            /* margin-bottom: 15px; */
            /* height: 42px; */
        }
        #toggleEmailBtn {
            padding-bottom: 0;
            margin-left: 0px;
            position: absolute;
            left: 0;
            /* right: 0; */
            line-height: 20px;
            /* bottom: -5px;     */
        }   
        .profile-principles-block {
            flex: 1 1 400px;
            min-width: 320px;
            background: #fff;
            border-radius: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.08);
            padding: 28px 20px 28px 20px;
            margin-bottom: 40px;
            max-width: 100%;
        }
        .principles-grid {
            display: flex;
            flex-flow:column;
            gap: 15px;
        }
        .principle-card {
            background: #f7f7f7;
            border-radius: 16px;
            min-height: 310px;
            /* height: 310px; */
            display: flex;
            flex-direction: column;
            justify-content: flex-start;
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
            padding: 18px 18px 18px 18px;
            position: relative;
            border: 1px solid lightgrey;
        }
        .principle-card.pending {
    margin-bottom: 10px;
}
        .principle-title {
            font-size: 1.15rem;
            color: #4a90e2;
            margin-bottom: 6px;
            font-weight: bold;
            overflow: hidden;
    word-wrap: break-word;
        }
        .principle-desc {
            width: 100%;
            color: #444;
            font-size: 0.75rem;
            margin-bottom: 8px;
            margin-top: 10px;
            flex: 1;
            padding-right: 0px;
            word-wrap: break-word;
        }
        .principle-content {
            display: flex;
            flex-direction: column;
            height: 100%;
            width: 100%;
        }
        .principle-meta {
            position: absolute;
            right:15px;
            color: #888;
            font-size: 0.65rem;
            margin-bottom: 4px;
            display: flex;
            justify-content: space-between;
            align-items: flex-end;
        }
        .principle-action-btn {
            padding: 0 5px;
            background: none;
            border: none;
            cursor: pointer;
            font-size: 0.95rem;
            color: #888;
            transition: color 0.2s;
        }
        
        .moderation-status {
            display: flex;
            align-items: center;
            gap: 6px;
            color: #e28f00;
            font-size: 1rem;
            margin-top: 8px;
            padding-top: 6px;
        }
        .empty-slot {
            background: none;
            border: 2px dotted #4a90e2;
            border-radius: 16px;
            min-height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #4a90e2;
            font-size: 1.1rem;
            flex-direction: column;
            gap: 8px;
            cursor: pointer;
            transition: background .2s;
        }
        
        .empty-slot .empty-icon {
            font-size: 2.2rem;
            margin-bottom: 4px;
            color: #4a90e2;
        }
        .placeholder-card {
            width: 100%;
            height: 100%;
            background: none;
            border: 2px solid #e0e0e0;
            border-radius: 16px;
            min-height: 180px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #aaa;
            font-size: 1.1rem;
            text-align: center;
            padding: 18px;
        }
        .principle-card.filled {
            background: #f7f7f7;
        }
        /* #profile-value-username span,#profile-value-joined span{
            margin-top: -10px;
    display: block;
        } */
        
        @media (max-width: 900px) {
            .profile-main {
                flex-direction: column;
                gap: 0;
                margin-top: 0px;

            }
            .profile-info-block {
                margin-bottom: 15px;
                padding-left: 15px;
            }
            .profile-principles-block {
                margin-bottom: 40px;
            }
            .profile-principles-block{
                padding: 30px 15px;
            }
            .principle-number-label{
                top: -25px !important;
            }
            .profile-value .updated-margin{
                /* margin-top: -27px; */
            }
            /* .profile-info-block{
                padding: 10px 10px 10px;
            } */
        }
        .profile-info-form input[readonly] {
            background: #f7f7f7;
            cursor: pointer;
        }
        .add-principle-btn {
            background: #f9a1b0;
            color: #fff;
            border: none;
            border-radius: 30px;
            padding: 12px 28px;
            font-size: 1.1rem;
            box-shadow: 0 2px 8px #f9a1b0;
            display: flex;
            align-items: center;
            gap: 10px;
            cursor: pointer;
            transition: background 0.2s;
        }
       
        .add-principle-btn .plus-icon {
            font-size: 1.4em;
            font-weight: bold;
        }
        .nav-container {
            width: 100%;
            max-width: 900px;
            margin: 0 auto;
        }
        .nav-left {
            display: flex;
            width: 100%;
            gap: 0;
        }
        .logout-btn {
            background: none;
            color: red;
            border: none;
            padding: 0 0 0 0px;
            font-size: 1.25rem;
            font-weight: bold;
            cursor: pointer;
            float: right;
            line-height: 17px;
            margin:0;
        }
        /* Firefox-specific line-height for logout button */
        @-moz-document url-prefix() {
            .logout-btn {
                line-height: 15px !important;
            }
        }
        /* Flexbox layout for profile info */
        .profile-info-table {
            width: 100%;
            margin-top: 5px;
        }
        .profile-row {
            display: -webkit-box;      /* Safari old syntax */
            display: -webkit-flex;     /* Safari new syntax */
            display: flex;
            -webkit-box-orient: vertical;
            -webkit-box-direction: normal;
            -webkit-flex-direction: column;
            flex-direction: column;
            gap: 0;
            /* margin-bottom: 7px; */
        }
        /* .profile-row.upperProfileBlockRow {
            flex-flow: row;
        } */
        .profile-labels-row {
            margin-bottom: 0;
        }
        /* .upperProfileBlockRow {
            margin-bottom: 15px;
        } */
        .profile-label {
            -webkit-box-flex: 1;
            -webkit-flex: 1;
            flex: 1;
            padding: 5px 0;
            font-size: 0.75rem;
            color: #555;
            font-weight: 900;
            text-align: left;
        }
        .profile-value {
            /* -webkit-box-flex: 1;
            -webkit-flex: 1;
            flex: 1; */
            height: 41px;
            font-size: 0.75rem;
            color: #333;
            -webkit-box-sizing: border-box;
            box-sizing: border-box;
            min-width: 0; /* Safari fix for flex overflow */
            /* text-align: center; */
            margin-bottom: 8px;
        }
        .profile-value strong{
            display:block;
            /* margin-top: 15px; */
            margin-bottom: 4px;
            font-size: 0.80rem;
        }
        #saveProfileBtn {
            line-height: 19px;
            padding-bottom: 0;
        }

        #profile-value-email .email-ellipsis{
            display: inline-block;
            vertical-align: bottom;
            /* Safari-specific fixes */
            -webkit-box-sizing: border-box;
            box-sizing: border-box;
            line-height: 8px;
        }
        /* Apply padding-bottom only when viewing other profiles */
        .viewing-other-profile .profile-info-block{
            height: unset;
        }
        .viewing-other-profile .profile-value.last__row__no__margin{
            height: 20px;
        }
        .edit-btn {
            background: none;
            border: none;
            cursor: pointer;
            font-size: 1.2rem;
            color: #888;
            transition: color 0.2s;
            /* display: none; //Hidden by default */
        }
        
        .profile-edit-input {
            width: 99%;
            padding: 4px;
            border-radius: 4px;
            border: 1px solid #eee;
            font-size: 0.65rem;
            /* text-align: center; */
            box-sizing: border-box; /* Include padding and border in element's total width and height */
        }
        .edit-tick, .edit-cross {
            background: none;
            border: none;
            cursor: pointer;
            font-size: 1.2rem;
            color: #888;
            transition: color 0.2s;
        }
        
        .responsive-table-wrapper {
            overflow-x: unset;
            -webkit-overflow-scrolling: unset; /* Smooth scrolling on iOS */
        }
        .last__row__no__margin{
            margin-bottom: 0;
        }
        
        @media (max-width: 400px) {
            .profile-info-block-inner{
                padding-right: 15px;
            }
           
            #toggleEmailBtn{
                font-size: 0.75rem !important;
                margin-top: -4px;
            }
        }
    </style>
</head>
<body class="profile-page <?php echo !$is_own_profile ? 'viewing-other-profile' : 'viewing-own-profile'; ?>">
<nav class="navbar">
    <div class="nav-container nav-flex">
        <div class="nav-left">
            <a href="index.php" class="nav-button"><?php echo t('prescription', $lang); ?></a>
            <a href="principles.php" class="nav-button"><?php echo t('principles', $lang); ?></a>
            <a href="walkers.php" class="nav-button"><?php echo t('walkers', $lang); ?></a>
            <?php if (isLoggedIn()): ?>
                <a href="profile.php" class="nav-button profile-btn"><?php echo t('profile', $lang); ?></a>
            <?php else: ?>
                <a href="go.php" class="nav-button login-btn"><?php echo t('login_signup_topbar', $lang); ?></a>
            <?php endif; ?>
        </div>
    </div>
</nav>
<div class="profile-main">
    <?php if ($success_message): ?>
        <div class="success-message" style="background-color: #e0f2f1; color: #00796b; padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: center; font-weight: bold;">
            <?php echo htmlspecialchars($success_message); ?>
        </div>
        <script>
            setTimeout(() => {
                const msg = document.querySelector('.success-message');
                if(msg) msg.style.display = 'none';
            }, 4000);
        </script>
    <?php endif; ?>
    <?php if ($update_message): ?>
        <div class="error-message" style="background-color: #ffebee; color: #c62828; padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: center; font-weight: bold;">
            <?php echo htmlspecialchars($update_message); ?>
        </div>
        <script>
            setTimeout(() => {
                const msg = document.querySelector('.error-message');
                if(msg) msg.style.display = 'none';
            }, 5000);
        </script>
    <?php endif; ?>
    <div class="profile-info-block">
        <div class="profile-info-block-inner">
            <form id="profileEditForm" method="post">
                <?php if ($is_own_profile): ?>
                    <div id="editProfileButtons">
                        <button type="button" id="editProfileBtn" class="profile-info-form edit-btn"><?php echo t('icon_edit', $lang); ?></button>
                        <button type="button" id="saveProfileBtn" class="profile-info-form save-btn" style="display:none;"><?php echo t('icon_save', $lang); ?></button>
                        <button type="submit" name="logout" class="logout-btn" title="<?php echo t('logout', $lang); ?>"><?php echo t('icon_logout', $lang); ?></button>
                        <button type="button" id="cancelProfileBtn" class="profile-info-form cancel-btn" style="display:none !important;"><?php echo t('icon_cancel', $lang); ?></button>
                    </div>
                <?php endif; ?>
                
                <div class="responsive-table-wrapper">
                    <div class="profile-info-table" id="profileFields">
                        <!-- First Data Row: Username and Joined Date -->
                        <div class="profile-row upperProfileBlockRow">
                            <div class="profile-value" id="profile-value-username">
                                <strong><?php echo t('username', $lang); ?></strong>
                                <span><?php echo htmlspecialchars($user['username']); ?></span>
                            </div>
                            <div class="profile-value" id="profile-value-joined">
                                <strong><?php echo t('joined', $lang); ?></strong>
                                <span><?php echo date('m/y', strtotime($user['joined'])); ?></span>
                            </div>
                        </div>
                        
                        <!-- Second Data Row: Email and Password -->
                        <div class="profile-row">
                            <div class="profile-value" id="profile-value-email">
                                <strong><?php echo t('email', $lang); ?></strong>
                                <span id="obfuscated-email" class="email-ellipsis" style="vertical-align: sub;"></span>
                                <button type="button" id="toggleEmailBtn" class="profile-info-form save-btn" style="font-size:0.95rem;">👁️</button>
                            </div>
                            <?php if ($is_own_profile): ?>
                                <div class="profile-value last__row__no__margin" id="profile-value-password">
                                    <strong class=""><?php echo t('password', $lang); ?></strong> ••••••
                                </div>
                            <?php else: ?>
                                <div class="profile-value last__row__no__margin"></div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>
    <div class="profile-principles-block" <?php 
        // For public profiles, filter out non-approved principles
        if (!$is_own_profile) {
            $user_principles_temp = array_values(array_filter($user_principles, function($p) {
                return isset($p['approved']) && $p['approved'] === true;
            }));
            $temp_count = count($user_principles_temp);
        } else {
            $temp_count = count($user_principles);
        }
        // Add inline style if principle count is 0 and own profile
        if ($temp_count === 0 && $is_own_profile) {
            echo 'style="flex: 1 1 100px;"';
        }
    ?>>
<?php 
// For public profiles, filter out non-approved principles
if (!$is_own_profile) {
    $user_principles = array_values(array_filter($user_principles, function($p) {
        return isset($p['approved']) && $p['approved'] === true;
    }));
}

$principle_count = count($user_principles);
?>
        <?php if ($is_own_profile): ?>
        <div class="principles-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; padding: 0 0px;">
            <div class="principles-counter" style="font-size: 0.65rem; color: #333; font-weight: bold;">
                <?php echo t('principles', $lang); ?>: <?php echo $principle_count; ?>/10
            </div>
            <?php if ($principle_count < 10): ?>
            <button class="plus-btn-full">
                            <span class="empty-icon">+</span>
                        </button>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        <div class="principles-grid" id="principlesGrid">
<?php 
// For profiles with no principles, show message
if ($principle_count === 0) {
    if ($is_own_profile) {
    } else {
        echo '<div style="text-align: center; padding: 40px; color: #888; font-size: 1.1rem;">';
        echo t('no_principles', $lang);
        echo '</div>';
    }
} else {
    // For own profile: show all principles, no empty cards
    // For public profile: only show approved principles, no empty cards
    if ($is_own_profile) {
        $cards_to_show = $principle_count; // Only show actual principles, no empty cards
    } else {
        $cards_to_show = $principle_count; // Only show actual approved principles
    }
    
    for ($i = 0; $i < $cards_to_show; $i++):
        $p = $user_principles[$i];
?>
    <?php if ($is_own_profile): ?>
        <div class="principle-card<?php echo empty($p['approved']) ? ' pending' : ''; ?>" data-slot="<?php echo $i; ?>">
            <div class="principle-content" data-id="<?php echo $p['id']; ?>">
                <div class="principle-header">
                    <div class="principle-title principle-title-view"> <?php echo htmlspecialchars($p['title']); ?> </div>
                    <div class="principle-meta">
                    <div>
                    <?php if ($is_own_profile && empty($p['approved'])): ?>
                        <span class="moderation-icon" title="Waiting for moderation">⏳</span>
                    <?php endif; ?>
                    <?php if ($is_own_profile): ?>
                        <button class="principle-action-btn edit-principle-btn" data-id="<?php echo $p['id']; ?>" title="Edit">✏️</button>
                        <form method="post" style="display:inline;">
                            <input type="hidden" name="principle_action" value="delete">
                            <input type="hidden" name="principle_id" value="<?php echo $p['id']; ?>">
                            <button type="submit" class="principle-action-btn" title="Delete" onclick="return confirm('Delete this principle?');">🗑️</button>
                        </form>
                    </div>
                    <?php endif; ?>
                    <span class="time-ago" title="<?php echo $p['created_at']; ?>"> <?php echo date('m/y', strtotime($p['created_at'])); ?> </span>
                    </div>
                </div>
                <div class="principle-desc principle-desc-view"> <?php echo nl2br(htmlspecialchars($p['description'])); ?> </div>
                <?php if ($is_own_profile): ?>
                    <form method="post" class="edit-principle-form ep__form" style="display:none;margin-top:0;">
                        <input type="hidden" name="principle_action" value="edit">
                        <input type="hidden" name="principle_id" value="<?php echo $p['id']; ?>">
                        <input type="text" name="edit_title" maxlength="100" placeholder="<?php 
                            $editTitleWordLimit = getSetting('PRINCIPLE_SETTINGS.title_word_limit', 1);
                            $editWordLabel = $editTitleWordLimit === 1 ? t('word', $lang) : t('words', $lang);
                            $editTitleLabel = $lang === 'other' ? getSetting('PRINCIPLE_SETTINGS.title_label_other', 'Principe') : getSetting('PRINCIPLE_SETTINGS.title_label_eng', 'Principle');
                            echo $editTitleLabel . ' (' . $editTitleWordLimit . ' ' . $editWordLabel . ')';
                        ?>" required class="edit-title-input" style="margin-bottom:6px;width:72%;display:inline;">
                        <span class="editTitleCount" style="display:block;margin:3px;color:#888;font-size:0.75em;float:right;">0/<?php echo getSetting('PRINCIPLE_SETTINGS.title_word_limit', 1); ?></span>
                        <textarea name="edit_desc" maxlength="<?php echo getSetting('PRINCIPLE_SETTINGS.description_char_limit', 777); ?>" placeholder="<?php 
                            $editDescLabel = $lang === 'other' ? getSetting('PRINCIPLE_SETTINGS.description_label_other', 'Description') : getSetting('PRINCIPLE_SETTINGS.description_label_eng', 'Description');
                            $editCharactersLabel = t('characters', $lang);
                            echo $editDescLabel . ' (' . getSetting('PRINCIPLE_SETTINGS.description_char_limit', 777) . ' ' . $editCharactersLabel . ')';
                        ?>" required class="edit-desc-input" style="width:100%;margin-top:5px;margin-bottom:6px;display:inline;min-height:315px;"></textarea>
                        <span class="editDescCount" style="display:block;width:100%;color:#888;font-size:0.75em;text-align:right;">0/<?php echo getSetting('PRINCIPLE_SETTINGS.description_char_limit', 777); ?></span>
                        <button type="submit" class="profile-info-form save-btn pencil__icon" style=""><?php echo t('icon_save', $lang); ?></button>
                        <button type="button" class="profile-info-form save-btn cancel-edit-btn"><?php echo t('icon_cancel', $lang); ?></button>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    <?php else: ?>
        <!-- Other users' profile: only show approved principles -->
        <div class="principle-card">
            <div class="principle-content" data-id="<?php echo $p['id']; ?>">
                <div class="principle-header">
                    <div class="principle-title principle-title-view"> <?php echo htmlspecialchars($p['title']); ?> </div>
                    <div class="principle-meta">
                        <span class="time-ago" title="<?php echo $p['created_at']; ?>"> <?php echo date('m/y', strtotime($p['created_at'])); ?> </span>
                    </div>
                </div>
                <div class="principle-desc principle-desc-view"> <?php echo nl2br(htmlspecialchars($p['description'])); ?> </div>
            </div>
        </div>
    <?php endif; ?>
<?php endfor; ?>
<?php } // End of else block for principles display ?>
</div>
</div>
<script>
// =============================================================================
// CLIENT-SIDE JAVASCRIPT - Profile Page Functionality
// =============================================================================

// Email reveal on click (legacy - currently unused)
const emailInput = document.getElementById('profile-email');
if (emailInput) {
    emailInput.addEventListener('click', function() {
        this.value = <?php echo json_encode($user['email']); ?>;
        this.style.cursor = 'default';
    }, { once: true });
}

// Character countdown for principle description (legacy - currently unused)
const descTextarea = document.getElementById('principle_desc');
const charCountEl = document.getElementById('charCount');
if (descTextarea && charCountEl) {
    const maxLength = descTextarea.maxLength;
    const updateCount = () => {
        const remaining = maxLength - descTextarea.value.length;
        charCountEl.textContent = `${remaining} characters remaining`;
    };
    descTextarea.addEventListener('input', updateCount);
    updateCount(); // Initial count
}

// Auto-hide success/error messages after 5 seconds (legacy - currently unused)
setTimeout(function() {
    var updateMsg = document.getElementById('updateMessage');
    if (updateMsg) updateMsg.style.display = 'none';
    var passwordMsg = document.getElementById('passwordUpdateMessage');
    if (passwordMsg && passwordMsg.style.color === 'green') passwordMsg.style.display = 'none';
}, 5000);

// Get principle limits from settings
const titleWordLimit = <?php echo getSetting('PRINCIPLE_SETTINGS.title_word_limit', 1); ?>;
const descCharLimit = <?php echo getSetting('PRINCIPLE_SETTINGS.description_char_limit', 777); ?>;

// Title word limit enforcement (legacy - for static forms, currently not used)
const titleInput = document.getElementById('principle_title');
const titleCount = document.getElementById('titleCount');
if (titleInput && titleCount) {
    titleInput.addEventListener('input', function() {
        // Normalize multiple spaces to single space (if enabled in settings)
        <?php if (getSetting('PRINCIPLE_SETTINGS.normalize_spaces', true)): ?>
        const cursorPos = this.selectionStart;
        const normalizedValue = this.value.replace(/\s+/g, ' ');
        if (normalizedValue !== this.value) {
            this.value = normalizedValue;
            this.setSelectionRange(cursorPos, cursorPos);
        }
        <?php endif; ?>
        
        // Count words and update display
        const words = this.value.trim().split(/\s+/).filter(word => word.length > 0);
        titleCount.textContent = `${words.length}/${titleWordLimit}`;
        
        // Enforce word limit by truncating
        if (words.length > titleWordLimit) {
            this.value = words.slice(0, titleWordLimit).join(' ');
            titleCount.textContent = `${titleWordLimit}/${titleWordLimit}`;
        }
    });
}



// =============================================================================
// PROFILE EDIT MODE - Toggle between view and edit states
// =============================================================================

const editProfileBtn = document.getElementById('editProfileBtn');
const profileFields = document.getElementById('profileFields');
const editProfileButtons = document.getElementById('editProfileButtons');
const logoutButton = document.getElementsByClassName('logout-btn');
let originalValues = {};  // Store original values for cancel functionality

// Handle edit button click - switch to edit mode
if (editProfileBtn) {
    editProfileBtn.addEventListener('click', function() {
        // Get real email from PHP (not the obfuscated display value)
        const realEmail = <?php echo json_encode($user['email']); ?>;
        
        // Save original values (extract only the span content, not the label)
        originalValues = {
            username: (document.querySelector('#profile-value-username span')?.textContent || document.getElementById('profile-value-username').textContent.replace(/^.*?\n/, '')).trim(),
            email: realEmail || '',
            joined: (document.querySelector('#profile-value-joined span')?.textContent || document.getElementById('profile-value-joined').textContent.replace(/^.*?\n/, '')).trim(),
            password: '••••••' // Keep password as is
        };
        // Extract clean email value
        let emailValue = originalValues.email;
        
        // Replace with two-row layout structure to match the original
        profileFields.innerHTML = `
            <!-- First Data Row: Username and Joined Date (read-only with labels) -->
            <div class='profile-row upperProfileBlockRow'>
                <div class='profile-value' id='profile-value-username'>
                    <strong><?php echo t('username', $lang); ?></strong>
                    ${originalValues.username}
                </div>
                <div class='profile-value' id='profile-value-joined'>
                    <strong><?php echo t('joined', $lang); ?></strong>
                    ${originalValues.joined}
                </div>
            </div>
            
            <!-- Second Data Row: Email and Password (editable with labels) -->
            <div class='profile-row'>
                <div class='profile-value' id='profile-value-email'>
                    <strong>Email</strong>
                    <input class="profile-edit-input" type='email' name='email' id='email-input' value='${emailValue}' maxlength='40'>
                </div>
                <div class='profile-value' id='profile-value-password'>
                    <strong>Password</strong>
                    <input class="profile-edit-input" type='password' name='new_password' id='new_password' placeholder='New password' style='margin-bottom:4px;'>
                    <input class="profile-edit-input" type='password' name='confirm_password' id='confirm_password' placeholder='Confirm new password'><br>
                    <div id='password_error' style='color:red;font-size:0.9em;margin-top:4px;display:none;'></div>
                </div>
            </div>
        `;
        // Add password-label-strong class to the password strong element
        const passwordStrong = document.querySelector('#profile-value-password strong');
        const emailStrong = document.querySelector('#profile-value-email strong');
        if (passwordStrong) {
            passwordStrong.classList.add('password-label-strong');
            emailStrong.classList.add('updated-margin');
        }
        
        // Add editing class to responsive-table-wrapper
        const tableWrapper = document.querySelector('.responsive-table-wrapper');
        if (tableWrapper) {
            tableWrapper.classList.add('editing');
        }
        
        // Hide edit button, show save/cancel buttons
        document.getElementById('editProfileBtn').style.display = 'none';
        document.getElementsByClassName('logout-btn')[0].style.display = 'none';
        document.getElementById('saveProfileBtn').style.display = 'inline-block';
        document.getElementById('cancelProfileBtn').style.display = 'none';
        
        // Handle save button click
        document.getElementById('saveProfileBtn').addEventListener('click', function(e) {
            const newPassword = document.getElementById('new_password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            if (newPassword && newPassword !== confirmPassword) {
                e.preventDefault();
                document.getElementById('password_error').textContent = '<?php echo t('password_mismatch', $lang); ?>';
                document.getElementById('password_error').style.display = 'block';
                return false;
            } else {
                // Passwords match or no password change, submit the form
                document.getElementById('password_error').style.display = 'none';
                
                // Remove password-label-strong class from the password strong element
                const passwordStrong = document.querySelector('#profile-value-password strong');
                if (passwordStrong) {
                    passwordStrong.classList.remove('password-label-strong');
                }
                
                // Find the profile edit form and submit it
                const form = document.getElementById('profileEditForm');
                if (form) {
                    form.submit();
                }
            }
        });
        // Add password confirmation validation
        const newPasswordField = document.getElementById('new_password');
        const confirmPasswordField = document.getElementById('confirm_password');
        const passwordError = document.getElementById('password_error');
        
        function validatePasswords() {
            const newPassword = newPasswordField.value;
            const confirmPassword = confirmPasswordField.value;
            
            if (confirmPassword && newPassword !== confirmPassword) {
                passwordError.textContent = '<?php echo t('password_mismatch', $lang); ?>';
                passwordError.style.display = 'block';
                return false;
            } else {
                passwordError.style.display = 'none';
                return true;
            }
        }
        
        newPasswordField.addEventListener('input', validatePasswords);
        confirmPasswordField.addEventListener('input', validatePasswords);
        
        // Prevent spaces in email field
        const emailInput = document.getElementById('email-input');
        if (emailInput) {
            emailInput.addEventListener('input', function(e) {
                // Remove any spaces from the email
                this.value = this.value.replace(/\s/g, '');
            });
            
            // Also prevent space key from being entered
            emailInput.addEventListener('keydown', function(e) {
                if (e.key === ' ' || e.keyCode === 32) {
                    e.preventDefault();
                    return false;
                }
            });
        }
        
        // Add cancel button functionality
        document.getElementById('cancelProfileBtn').addEventListener('click', function() {
            // Remove editing class from responsive-table-wrapper
            const tableWrapper = document.querySelector('.responsive-table-wrapper');
            if (tableWrapper) {
                tableWrapper.classList.remove('editing');
            }
            
            // Restore original view
            profileFields.innerHTML = `
                <!-- First Data Row: Username and Joined Date with labels -->
                <div class='profile-row upperProfileBlockRow'>
                    <div class='profile-value' id='profile-value-username'>
                        <strong><?php echo t('username', $lang); ?></strong>
                        <span>${originalValues.username}</span>
                    </div>
                    <div class='profile-value' id='profile-value-joined'>
                        <strong><?php echo t('joined', $lang); ?></strong>
                        <span>${originalValues.joined}</span>
                    </div>
                </div>
                
                <!-- Second Data Row: Email and Password with labels -->
                <div class='profile-row'>
                    <div class='profile-value' id='profile-value-email'>
                        <strong>Email</strong>
                        <span id="obfuscated-email" class="email-ellipsis" style="vertical-align: sub;"></span>
                        <button type="button" id="toggleEmailBtn" class="profile-info-form save-btn" style="font-size:0.95rem;">👁️</button>
                    </div>
                    <div class='profile-value' id='profile-value-password'>
                        <strong>Password</strong> ••••••
                    </div>
                </div>
            `;
            
            // Remove password-label-strong class from the password strong element
            const passwordStrong = document.querySelector('#profile-value-password strong');
            if (passwordStrong) {
                passwordStrong.classList.remove('password-label-strong');
            }
            
            // Re-initialize email toggle button after cancel
            const toggleEmailBtnRestore = document.getElementById('toggleEmailBtn');
            const obfuscatedEmailRestore = document.getElementById('obfuscated-email');
            if (toggleEmailBtnRestore && obfuscatedEmailRestore) {
                const realEmail = <?php echo json_encode($user['email']); ?>;
                toggleEmailBtnRestore.addEventListener('click', function() {
                    if (realEmail && realEmail.trim() !== '') {
                        obfuscatedEmailRestore.textContent = realEmail;
                        toggleEmailBtnRestore.style.display = 'none';
                    } else {
                        obfuscatedEmailRestore.textContent = '';
                        toggleEmailBtnRestore.style.display = 'none';
                    }
                });
            }
            
            // Show edit button, hide save/cancel buttons
            document.getElementById('editProfileBtn').style.display = 'inline-block';
            document.getElementById('saveProfileBtn').style.display = 'none';
            document.getElementById('cancelProfileBtn').style.display = 'none';
        });
    });
}
// =============================================================================
// EMAIL REVEAL FUNCTIONALITY - Show/hide email with eye icon
// =============================================================================

// Handle email reveal button click (works for own profile and others' profiles)
const toggleEmailBtn = document.getElementById('toggleEmailBtn');
const obfuscatedEmail = document.getElementById('obfuscated-email');
if (toggleEmailBtn && obfuscatedEmail) {
    const realEmail = <?php echo json_encode($user['email']); ?>;
    toggleEmailBtn.addEventListener('click', function() {
        if (realEmail && realEmail.trim() !== '') {
            // Show the real email
            obfuscatedEmail.textContent = realEmail;
            toggleEmailBtn.style.display = 'none';  // Hide the eye icon after revealing
        } else {
            // No email to show
            obfuscatedEmail.textContent = '';
            toggleEmailBtn.style.display = 'none';
        }
    });
}

// =============================================================================
// ADD PRINCIPLE FORM - Dynamically create form when plus button is clicked
// =============================================================================

// Handle plus button click to show add principle form
document.querySelectorAll('.plus-btn-full').forEach(btn => {
    btn.addEventListener('click', function(e) {
        e.preventDefault();
        
        // Prevent creating multiple forms if button already clicked
        if (btn.disabled) return;
        
        // Disable the plus button to prevent multiple submissions
        btn.disabled = true;
        btn.style.opacity = '0.5';
        btn.style.pointerEvents = 'none';
        
        // Get customizable labels from settings (supports bilingual)
        const titleLabel = '<?php echo $lang === 'other' ? getSetting('PRINCIPLE_SETTINGS.title_label_other', 'Principe') : getSetting('PRINCIPLE_SETTINGS.title_label_eng', 'Principle'); ?>';
        const descLabel = '<?php echo $lang === 'other' ? getSetting('PRINCIPLE_SETTINGS.description_label_other', 'Description') : getSetting('PRINCIPLE_SETTINGS.description_label_eng', 'Description'); ?>';
        const wordLabel = titleWordLimit === 1 ? '<?php echo t('word', $lang); ?>' : '<?php echo t('words', $lang); ?>';
        const charactersLabel = '<?php echo t('characters', $lang); ?>';
        
        // Create the container and form HTML
        const container = document.createElement('div');
        container.className = 'principle-card';
        container.innerHTML = `
            <div class="empty-slot-container">
                <form method="post" action="" class="add-principle-form np__form" autocomplete="off">
                    <input type="hidden" name="add_principle" value="1">
                    <div class="add-principle-title">
                        <input type="text" name="principle_title" maxlength="100" placeholder="${titleLabel} (${titleWordLimit} ${wordLabel})" required id="addPrincipleTitle">
                        <span id="addTitleCount" style="display:block;margin:3px;color:#888;font-size:0.75em;float:right;">0/${titleWordLimit}</span>
                    </div>
                    <div class="add-principle-desc">
                        <textarea name="principle_desc" maxlength="${descCharLimit}" placeholder="${descLabel} (${descCharLimit} ${charactersLabel})" required id="addPrincipleDesc"></textarea>
                        <span id="addDescCount" style="display:block;width:100%;color:#888;font-size:0.75rem;text-align:right;">0/${descCharLimit}</span>
                    </div>
                    <button type="submit" class="profile-info-form save-btn pencil__icon"><?php echo t('icon_save', $lang); ?></button>
                </form>
            </div>
        `;
        
        // Find the principle grid and insert the form container
        const principleGrid = document.getElementById('principlesGrid');
        if (principleGrid) {
            // Hide the "no principles" message if it exists
            const noPrinciplesMessage = principleGrid.querySelector('div[style*="text-align: justify"]');
            if (noPrinciplesMessage) {
                noPrinciplesMessage.style.display = 'none';
            }
            
            // Insert at the top so the new form appears at the top of the grid
            principleGrid.insertBefore(container, principleGrid.firstChild);
            
            // Automatically focus the title input for better UX
            const titleInput = container.querySelector('input[name="principle_title"]');
            if (titleInput) {
                titleInput.focus();
            }
            
            // Hide the plus button when form is submitted (will reappear after page reload)
            const addForm = container.querySelector('form.add-principle-form');
            if (addForm) {
                addForm.addEventListener('submit', function() {
                    btn.style.display = 'none';
                });
            }
        }
    });
});

// =============================================================================
// REAL-TIME INPUT VALIDATION - For dynamically created principle forms
// =============================================================================

// Use event delegation to handle input validation for dynamically created forms
document.addEventListener('input', function(e) {
    // TITLE INPUT VALIDATION
    if (e.target.name === 'principle_title') {
        // Normalize spaces if enabled (replace multiple spaces with single space)
        <?php if (getSetting('PRINCIPLE_SETTINGS.normalize_spaces', true)): ?>
        const cursorPos = e.target.selectionStart;
        const normalizedValue = e.target.value.replace(/\s+/g, ' ');
        if (normalizedValue !== e.target.value) {
            e.target.value = normalizedValue;
            e.target.setSelectionRange(cursorPos, cursorPos);  // Preserve cursor position
        }
        <?php endif; ?>
        
        // Count words and enforce word limit
        const words = e.target.value.trim().split(/\s+/).filter(word => word.length > 0);
        const countSpan = e.target.parentElement.querySelector('span');
        if (countSpan) {
            countSpan.textContent = `${words.length}/${titleWordLimit}`;
            
            // Truncate if over limit
            if (words.length > titleWordLimit) {
                e.target.value = words.slice(0, titleWordLimit).join(' ');
                countSpan.textContent = `${titleWordLimit}/${titleWordLimit}`;
            }
        }
    }
    
    // DESCRIPTION INPUT VALIDATION
    if (e.target.name === 'principle_desc') {
        // Normalize spaces if enabled (replace multiple spaces with single space)
        <?php if (getSetting('PRINCIPLE_SETTINGS.normalize_spaces', true)): ?>
        const cursorPos = e.target.selectionStart;
        const normalizedValue = e.target.value.replace(/\s+/g, ' ');
        if (normalizedValue !== e.target.value) {
            e.target.value = normalizedValue;
            e.target.setSelectionRange(cursorPos, cursorPos);  // Preserve cursor position
        }
        <?php endif; ?>
        
        // Remove line breaks (principles don't support multi-line descriptions)
        e.target.value = e.target.value.replace(/[\r\n]+/g, ' ');
        
        // Count characters and enforce character limit
        const currentLength = e.target.value.length;
        const countSpan = e.target.parentElement.querySelector('span');
        if (countSpan) {
            countSpan.textContent = currentLength + '/' + descCharLimit;
            
            // Truncate if over limit
            if (e.target.value.length > descCharLimit) {
                e.target.value = e.target.value.slice(0, descCharLimit);
                countSpan.textContent = descCharLimit + '/' + descCharLimit;
            }
        }
    }
});

// Prevent Enter key from creating line breaks in description fields
document.addEventListener('keydown', function(e) {
    if (e.target.name === 'principle_desc' && e.key === 'Enter') {
        e.preventDefault();  // Block Enter key to prevent line breaks
    }
});


// =============================================================================
// INLINE EDIT PRINCIPLE - Edit existing principles within the card
// =============================================================================

/**
 * Setup inline editing for existing principles
 * 
 * Allows users to edit their principles without leaving the page.
 * Handles the transition between view mode and edit mode within each principle card.
 */
function setupInlineEditPrinciple() {
    document.querySelectorAll('.edit-principle-btn').forEach(function(btn) {
        const content = btn.closest('.principle-content');
        const titleView = content.querySelector('.principle-title-view');
        const descView = content.querySelector('.principle-desc-view');
        const form = content.querySelector('.edit-principle-form');
        const header = btn.closest(".principle-header");
        
        // Skip if elements not found
        if (!titleView || !descView || !form) return;
        
        // Handle edit button click
        btn.onclick = function(e) {
            e.preventDefault();
            
            // Expand header if present
            if (header) {
                header.classList.add("expanded");
            }
            
            // Save original text for cancel functionality
            const origTitle = titleView.textContent.trim();
            const origDesc = descView.textContent.trim();
            
            // Hide view elements, show edit form
            titleView.style.display = 'none';
            descView.style.display = 'none';
            form.style.display = 'block';
            
            // Place form inline (replace the text nodes)
            titleView.parentNode.insertBefore(form, titleView.nextSibling);
            
            // Pre-fill form fields with current values
            form.querySelector('.edit-title-input').value = origTitle;
            form.querySelector('.edit-desc-input').value = origDesc;
            
            // Setup title word limit validation
            const editTitleInput = form.querySelector('.edit-title-input');
            const editTitleCount = form.querySelector('.editTitleCount');
            
            // Initialize word count display
            const initialWords = editTitleInput.value.trim().split(/\s+/).filter(word => word.length > 0);
            editTitleCount.textContent = `${initialWords.length}/${titleWordLimit}`;
            
            // Real-time word limit enforcement for title
            editTitleInput.addEventListener('input', function() {
                const words = this.value.trim().split(/\s+/).filter(word => word.length > 0);
                editTitleCount.textContent = `${words.length}/${titleWordLimit}`;
                
                // Truncate if over limit
                if (words.length > titleWordLimit) {
                    this.value = words.slice(0, titleWordLimit).join(' ');
                    editTitleCount.textContent = `${titleWordLimit}/${titleWordLimit}`;
                }
            });
            
            // Setup description character limit validation
            const editDescInput = form.querySelector('.edit-desc-input');
            const editDescCount = form.querySelector('.editDescCount');
            
            // Real-time character limit enforcement for description
            editDescInput.addEventListener('input', function() {
                // Remove line breaks (not supported in principles)
                this.value = this.value.replace(/[\r\n]+/g, ' ');
                const currentLength = this.value.length;
                editDescCount.textContent = currentLength + '/777';
                
                // Truncate if over limit
                if (this.value.length > 777) {
                    this.value = this.value.slice(0,777);
                    editDescCount.textContent = '777/777';
                }
            });
            
            // Prevent Enter key from creating line breaks in description
            editDescInput.addEventListener('keydown', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                }
            });
            
            // Initialize character counter
            editDescCount.textContent = editDescInput.value.length + '/777';
            
            // Handle cancel button - restore view mode
            form.querySelector('.cancel-edit-btn').onclick = function() {
                form.style.display = 'none';
                titleView.style.display = '';
                descView.style.display = '';
                
                // Collapse header if present
                if (header) {
                    header.classList.remove("expanded");
                }
            };
        };
    });
}

// Initialize inline editing for all existing principles
setupInlineEditPrinciple();

</script>
</body>
</html> 