<?php
// Enable error reporting for debugging (can be disabled in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Load required configuration and utility files
require_once 'functions.php';

// =============================================================================
// LOGOUT HANDLING
// =============================================================================

// Handle logout POST request (destroys session and redirects to homepage)
if (isset($_POST['logout'])) {
    logoutUser();
    header('Location: index.php');
    exit;
}

// =============================================================================
// USER MODERATION ACTIONS (Moderators Only)
// =============================================================================

// Handle user approval/rejection actions
// Only moderators can access this functionality
if (isModerator() && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['user_id'], $_POST['action'])) {
    $user_id = $_POST['user_id'];
    $action = $_POST['action'];  // Either 'approve' or 'deny'
    $all_users = getUsers();
    $found = false;

    // Search for the user and update their status
    foreach ($all_users as &$u) {
        if ($u['user_id'] === $user_id) {
            if ($action === 'approve') {
                $u['approved'] = true;  // Mark user as approved
            }
            $found = true;
            break;
        }
    }

    // If action is 'deny', remove the user entirely
    if ($found && $action === 'deny') {
        $all_users = array_filter($all_users, function ($u) use ($user_id) {
            return $u['user_id'] !== $user_id;  // Keep all except the denied one
        });
        $all_users = array_values($all_users);  // Re-index array
    }

    // Save updated users list
    saveUsers($all_users);
    
    // Redirect to refresh the page and show updated list
    header('Location: walkers.php?page=' . ($_GET['page'] ?? 1));
    exit;
}

// =============================================================================
// LANGUAGE AND USER DATA SETUP
// =============================================================================

// Get current language (auto-detect from browser or use session)
$lang = get_default_lang();

// Load all users from walkers.json and filter out 'admin' user
// Admin user should not appear in the public walkers list
$all_users = array_filter(getUsers(), function($u) { 
    return $u['username'] !== 'admin'; 
});

// =============================================================================
// FILTER USERS FOR MODERATION (Pending and Approved)
// =============================================================================

// Get pending users (only for moderators)
// These are users waiting for approval, shown at the top of the page
$pending_users = [];
if (isModerator()) {
    $pending_users = array_filter($all_users, function ($u) {
        // Filter criteria: Not approved (pending moderation)
        return empty($u['approved']);
    });
    // Sort pending users by join date (newest first)
    usort($pending_users, function($a, $b) {
        $ta = isset($a['joined']) ? strtotime($a['joined']) : 0;
        $tb = isset($b['joined']) ? strtotime($b['joined']) : 0;
        return $tb <=> $ta;
    });
}

// Get approved users (visible to everyone)
// These are the publicly visible users that passed moderation
$approved_users = array_filter($all_users, function($u) {
    return !empty($u['approved']);  // Must be approved
});
// Sort approved users by join date (newest first)
usort($approved_users, function($a, $b) {
    $ta = isset($a['joined']) ? strtotime($a['joined']) : 0;
    $tb = isset($b['joined']) ? strtotime($b['joined']) : 0;
    return $tb <=> $ta;
});

// =============================================================================
// PAGINATION SETUP (All Users - Pending + Approved)
// =============================================================================

// Get pagination settings and calculate page data
$users_per_page = getSetting('PAGES.walkers.walkers_per_page', 10);     // Number of walkers per page
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;      // Current page (minimum 1)

// Combine all users for pagination (pending first, then approved)
$all_users_for_pagination = array_merge($pending_users, $approved_users);
$total_users = count($all_users_for_pagination);                        // Total number of all users
$total_pages = ceil($total_users / $users_per_page);                    // Calculate total pages
$offset = ($page - 1) * $users_per_page;                                // Starting index for current page

// Slice the combined array to show only users for the current page
$paginated_users = array_slice($all_users_for_pagination, $offset, $users_per_page);

// Separate paginated users back into pending and approved for display
$pending_users = array_filter($paginated_users, function($u) { return empty($u['approved']); });
$approved_users = array_filter($paginated_users, function($u) { return !empty($u['approved']); });
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Walkers</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:ital,wght@0,100..900;1,100..900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="style.css">
    <link rel="stylesheet" href="style.php">
    <style>
        body { background: #e3e6f5; }
        
        @media screen and (min-width: 768px) {
            .walkers-page{
                /* padding-right: 15px; */
            }
        }
        
        .walkers-container {
            max-width: 400px;
            margin: 10px auto 40px auto;
            background: #fff3d6;
            border-radius: 30px;
            box-shadow: <?php echo getSetting('PAGES.walkers.container_box_shadow', '0 4px 15px rgba(0,0,0,0.1)'); ?>;
            padding: 15px 15px 25px 15px;
        }
        .walkers-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0 5px;
        }
        .walkers-table th, .walkers-table td {
            padding: 12.5px 10px;
            background: #fff;
            font-size: 0.75rem;
            color: #333;
            text-align: center;
        }
        .walkers-table th {
            background: #8fd3ff;
            color: #fff;
            font-size: 0.75rem;
        }
        .walkers-table th a {
            color: #fff;
            text-decoration: none;
            /* display: block; */
            /* transition: background-color 0.2s; */
        }
        .walkers-table thead tr th:first-child {
            border-top-left-radius: 12px;
            border-bottom-left-radius: 12px;
        }

        .walkers-table thead tr th:nth-child(2) {
            border-top-right-radius: 12px;
            border-bottom-right-radius: 12px;
        }
        
        .walkers-table tbody tr td:first-child {
            border-top-left-radius: 12px;
            border-bottom-left-radius: 12px;
        }

        .walkers-table tbody tr td:nth-child(2) {
            width: 55%;
            border-top-right-radius: 12px;
            border-bottom-right-radius: 12px;
        }
        .walker-username-link {
            color: #ffffff;
            text-decoration: underline;
            font-weight: bold;
            cursor: pointer;
        }
        .nav-container {
            width: 100%;
            max-width: 900px;
            margin: 0 auto;
        }
        .nav-left{
            display: flex;
            width: 100%;
            gap: 0;
        }
        .responsive-table-wrapper {
            overflow-x: auto;
        }

        @media screen and (max-width: 900px) {
            .walkers-container{
                margin-top: 0;
            }
        }
        @media screen and (max-width: 768px) {
            .walkers-container{
                padding: 15px;
                margin: 0 10px 40px;
            }
            .walkers-table th,.walkers-table td{
                font-size: 0.75rem;
            }
        }
        /* Pagination styles (customizable) */
        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 15px;
        }
        .pagination .pagination-btn {
            background: <?php echo getSetting('PAGES.walkers.pagination.button_bg', getSetting('COLORS.navbar_button_bg_color', '#8FD3FF')); ?>;
            color: <?php echo getSetting('PAGES.walkers.pagination.button_text', getSetting('COLORS.navbar_button_text_color', '#ffffff')); ?>;
        }
    </style>
</head>
<body class="walkers-page">
<nav class="navbar">
    <div class="nav-container nav-flex">
        <div class="nav-left">
            <a href="index.php" class="nav-button"><?php echo t('prescription', $lang); ?></a>
            <a href="principles.php" class="nav-button"><?php echo t('principles', $lang); ?></a>
            <a href="walkers.php" class="nav-button"><?php echo t('walkers', $lang); ?></a>
            <?php if (isLoggedIn()): ?>
                <a href="profile.php" class="nav-button profile-btn"><?php echo t('profile', $lang); ?></a>
            <?php else: ?>
                <a href="go.php" class="nav-button login-btn"><?php echo t('login_signup_topbar', $lang); ?></a>
            <?php endif; ?>
        </div>
    </div>
</nav>
<div class="walkers-container">
    <!-- Walkers Table (Pending + Approved) -->
    <div class="responsive-table-wrapper">
        <table class="walkers-table">
            <tbody>
            <?php if (empty($pending_users) && empty($approved_users)): ?>
                <tr>
                    <td colspan="2" style="text-align:center; padding: 40px; font-size: 1.2rem; color: #888;">
                        No walkers have joined yet. Be the first!
                    </td>
                </tr>
            <?php else: ?>
                <?php 
                // Combine pending and approved for display
                $all_display_users = array_merge($pending_users, $approved_users);
                foreach ($all_display_users as $user): 
                    $is_pending = empty($user['approved']);
                ?>
                    <tr>
                        <td>
                            <a href="profile?user=<?php echo urlencode($user['username']); ?>" class="walker-username-link">
                                <?php echo htmlspecialchars($user['username']); ?>
                            </a>
                        </td>
                        <td>
                            <?php if ($is_pending && isModerator()): ?>
                                <!-- Moderation buttons for pending users (moderators only) -->
                                <div style="display: flex; gap: 8px; justify-content: center;">
                                    <form method="post" style="display: inline-block; margin: 0;">
                                        <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($user['user_id']); ?>">
                                        <input type="hidden" name="action" value="approve">
                                        <button type="submit" style="background: #2ecc40;width:25px;height:25px; color: white; border: none; padding:0; border-radius: 50%; cursor: pointer; font-size: 1.3rem; font-weight: bold;" title="Approve User">
                                            <svg width="18" height="18" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M5 10.5L9 14.5L15 7.5" stroke="white" stroke-width="2.2" stroke-linecap="round" stroke-linejoin="round"/>
                                            </svg>
                                        </button>
                                    </form>
                                    <form method="post" style="display: inline-block; margin: 0;">
                                        <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($user['user_id']); ?>">
                                        <input type="hidden" name="action" value="deny">
                                        <button type="submit" style="background: #b71c1c;width:25px;height:25px; color: white; border: none; padding:0; border-radius: 50%; cursor: pointer; font-size: 1.3rem; font-weight: bold;" title="Reject User" onclick="return confirm('Are you sure you want to reject this user? This action cannot be undone.');">
                                            <svg width="16" height="16" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M6 6L14 14M14 6L6 14" stroke="white" stroke-width="2.2" stroke-linecap="round"/>
                                            </svg>
                                        </button>
                                    </form>
                                </div>
                            <?php elseif (!$is_pending): ?>
                                <!-- Show join date for approved users -->
                                <?php echo date('m/y', strtotime($user['joined'])); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <?php if ($total_pages > 1): ?>
    <?php
        $disabled_bg = getSetting('PAGES.walkers.pagination.button_bg_disabled', '#ccc');
        $disabled_text = getSetting('PAGES.walkers.pagination.button_text_disabled', '#999');
    ?>
    <div class="pagination">
        <a href="?page=<?php echo ($page - 1); ?>&lang=<?php echo $lang; ?>" class="pagination-btn" <?php if ($page <= 1) echo 'style="pointer-events:none;background:'.$disabled_bg.';color:'.$disabled_text.';"'; ?>>&#8592;</a>
        <a href="?page=<?php echo ($page + 1); ?>&lang=<?php echo $lang; ?>" class="pagination-btn" <?php if ($page >= $total_pages) echo 'style="pointer-events:none;background:'.$disabled_bg.';color:'.$disabled_text.';"'; ?>>&#8594;</a>
    </div>
    <?php endif; ?>
</div>

</body>
</html> 